rng = Random.default_rng()
Random.seed!(99)

#No Noise forecasting from eta = 0 to the 90th element of the domain etaspan = (0.05, 5.325)


#Constants
C = 0.01


#Initial Conditions
I = [1.0, 0.0]   #Psi(0)=1, Psi'(0)=1
etaspan = (0.05, 5.325)

#radius range
datasize= 100
etasteps = range(etaspan[1], etaspan[2]; length = datasize)


function whitedwarf(du, u, p, r)
    psi = u[1]
    dpsi = u[2]
    du[1] = dpsi
    du[2] = (-((psi^2-C))^(3/2) - 2/r * dpsi)
end


#Defining the Ordinary differential equation as an ODEProblem with the DifferentialEquations.jl
prob = ODEProblem(whitedwarf, I, etaspan)
#Solving the ODEProblem with the Tsit5() algorithm
sol = solve(prob,saveat=etasteps)

dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)

etasteps =  etasteps[1:end-80]
etaspan = (etasteps[1], etasteps[end])
I=[1.0,0.0]
prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end
#Training data
true_data= Array(sol[:,1:end-80])
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained_nonoise = (layer_1 = (weight = Float32[0.16588385 0.15334097; 0.03018726 -0.07110723; -0.23268887 -0.06052286; 0.049007546 -0.09596442; 0.32349116 0.341041; 0.2024078 -0.24246441; -0.519039 -0.23345262; 0.34465483 -0.12434384; -0.63985085 -0.3575339; -0.18219598 -0.12985368; -0.3050018 -0.053197447; 0.046625957 0.036069255; 0.070705466 0.05953196; 0.132027 -0.07635857; 0.032602232 0.05534347; -0.17325726 0.086415075; -0.29297772 -0.026925761; 0.2758424 -0.09552776; -0.041351546 -0.23493549; -0.10574097 -0.112423584; -0.31577575 -0.025024518; 0.3681532 0.19365692; -0.26275453 0.22619334; -0.30943292 -0.14927118; 0.3908735 0.3652251; -0.3565252 -0.04289322; 0.25531897 0.104206055; -0.0310991 0.09503806; 0.24841496 0.21356499; 0.42879504 0.1451434; -0.7521586 -0.07206575; -0.035593133 -0.18143517; -0.08842734 0.0023348283; -0.877373 -0.041289344; -0.1322577 -0.036330577; -0.3218336 -0.18301192; 0.42651546 0.21757068; 0.029464662 -0.0520659; 0.2822587 -0.022361604; -0.17184024 0.010806201; -0.20133461 -0.05991704; 0.12471187 -0.02613341; -0.3174329 -0.1711396; 0.15418805 -0.011544338; -0.13157462 0.1732229; -0.22733343 -0.0025040149; -0.1529283 0.00850684; 0.32756355 0.01609584; -0.09448479 -0.2235555; 0.10152629 0.026678596; -0.2641044 0.024074076; 0.02820846 -0.010281039; -0.040697627 -0.065487474; -0.11946053 0.009839505; 0.08616206 -0.12753253; 0.32491496 0.10914712; 0.60354453 0.26045847; 0.03723341 0.17430502; -0.38947612 0.004451308; -0.0437732 0.3141642; -0.13912529 0.056640755; 0.3361139 0.23522972; 0.0154023925 -0.008005405; -0.40386075 -0.043251336; 0.48087695 -0.18185355; 0.13795504 0.14496632; 0.34122562 0.12182374; -0.11859388 0.070387915; -0.17458837 -0.12792547; 0.067643665 0.22771533; 0.15623242 0.2437792; -0.02709819 0.35150763; -0.080195025 -0.049091365; 0.027172318 -0.27736792; 0.08013533 -0.088501774; -0.21622261 -0.13085993; -0.10310246 0.051254775; -0.13793811 0.03288491; 0.024785582 0.024959829; 0.6434003 0.19766387], bias = Float32[-0.16933082; 0.61738044; 0.2640265; -0.36669767; -0.4541679; 0.36637858; 0.39046997; 0.3053318; 0.14784575; 0.34012392; 0.15317003; 0.091653205; -0.3423197; -0.19105935; 0.06651548; -0.49515396; 0.20958638; 0.03087455; 0.33249474; 0.0490695; 0.004308433; -0.14627354; 0.158854; 0.35380042; -0.12446167; 0.5224705; -0.0837468; 0.4307604; -0.13038962; -0.22346681; -0.41910574; -0.061137665; 0.31534642; -0.11760677; -0.31582898; 0.4682405; -0.031384073; 0.53836083; -0.1581926; -0.4457118; 0.1861316; -0.07834268; 0.24355999; -0.30532566; 0.26316443; 0.135603; 0.42232096; 0.062274475; 0.45563254; -0.3137169; 0.43623626; 0.1292378; -0.39965704; 0.0034260112; 0.57286245; -0.10064936; 0.11106987; -0.15703665; -0.37400353; 0.053366944; 0.21044275; -0.31190705; -0.21992628; 0.28845292; 0.14413348; -0.1598821; -0.2951883; -0.4590125; 0.33344138; -0.38638344; -0.3152303; -0.33957714; 0.39353007; 0.6041802; -0.2638021; 0.19859694; -0.22682215; 0.51502776; -0.3629671; -0.06822401]), layer_2 = (weight = Float32[-0.07080636 0.20312144 0.011429686 -0.13104828 -0.19640806 0.32779217 -0.011738689 -0.15795693 0.1847635 0.105376564 -0.06402981 0.09294415 -0.11384765 0.09157849 0.13467821 -0.014200238 -0.08971435 0.16672039 0.1104113 0.13098055 0.1401367 -0.07814057 -0.13435492 0.16752854 -0.32497433 0.07866973 -0.13006768 -0.051553875 -0.0028434282 -0.18462968 0.21183342 0.16099109 0.08729984 -0.0024806415 -0.052414972 0.19791532 -0.074553706 -0.076378666 -0.007225612 -0.13165735 -0.113840826 0.012642994 0.004786522 0.19462164 -0.15279397 -0.0065468475 -0.055912722 -0.18801083 0.21827115 -0.10646357 -0.06896239 0.21944472 -0.12002363 -0.081720494 0.13857828 0.02660264 0.12385721 -0.11942253 0.043524373 -0.21768339 -0.1378343 -0.15427965 -0.11881333 0.07428945 -0.0062075467 -0.12647359 0.011268081 -0.100392 0.17586191 0.041381855 -0.20050967 -0.15198627 0.3149001 -0.028957004 0.024633287 -0.041725326 -0.041647173 0.0868211 -0.08035301 0.016162533; 0.003757035 0.030284809 -0.008461586 -0.069726385 -0.33410263 0.2946433 0.055547927 -0.1120976 -0.057501636 0.12485599 -0.102347456 0.06860336 0.0036713795 0.117666155 0.17252861 0.070023395 0.21696706 0.06134456 0.12658033 -0.0101769725 0.08198475 0.116866775 -0.036325887 0.01160169 -0.29096085 0.07037881 -0.030341966 -0.007415127 0.12506934 -0.17746285 0.098698735 0.14868298 0.10528811 0.0015329864 0.013678926 0.051794603 0.10749667 -0.0018189179 -0.040177774 -0.09879112 0.0398796 0.18295553 -0.028816432 0.15581301 -0.03546323 -0.04394993 0.073064744 -0.008178114 0.115950935 -0.09592301 -0.0673784 0.018992795 0.023719177 0.013155382 0.0022603183 0.042073745 -0.20503062 -0.17394109 -0.08203202 0.02433762 -0.11780772 -0.17606969 -0.08950082 0.22915208 -0.10178248 -0.12613356 0.10518538 0.05583288 0.2538676 -0.30632928 -0.23756847 0.03063563 0.18647766 0.042409364 0.011769269 -0.029071843 0.18074985 0.18226495 -0.21534012 -0.20851433; 0.0475457 0.1487505 0.022649758 0.019463476 -0.061366946 0.14328378 0.18756738 -0.03603678 -0.075399905 0.13130826 0.01249087 0.018232394 -0.031137317 -0.102817446 -0.15540022 -0.1501923 -0.14592853 -0.11376485 0.11405927 -0.06916056 -0.21891569 -0.0027347635 0.005804185 0.0134005295 -0.2683521 0.10304846 -0.11597897 0.021754568 -0.1124574 0.019242188 -0.12493624 -0.10500412 0.050030768 -0.17240757 -0.12309578 -0.19648914 -0.09088354 -0.18961436 0.08472296 0.030121874 0.10468831 0.0701886 0.31376433 -0.034429245 -0.02693739 -0.049116164 -0.18059182 -0.0056368955 -0.0056885825 -0.09183573 -0.0030768018 0.15371385 0.011019676 -0.20649111 -0.21229221 0.015482169 0.07707874 0.028698998 0.12779889 -0.12633328 0.07345171 0.16853316 0.14546433 0.047861107 -0.008512467 -0.014595938 -0.014746727 0.06976811 -0.04473106 -0.2343229 -0.023437599 -0.18494745 0.21655327 0.013512564 -0.087557115 0.15684195 0.16485095 0.02002786 0.017917568 0.15270138; 0.14515717 -0.26516336 0.08454019 0.11742598 0.2021465 -0.15127991 -0.42799392 -0.015885992 -0.38905054 -0.08801308 -0.13856754 -0.14109027 0.18614684 -0.12309659 -0.2590706 0.051926333 0.017598249 -0.07168667 -0.1424897 -0.0042025405 -0.015880289 0.06273468 -0.14347313 -0.17231396 0.046481747 -0.33832833 0.14522807 0.058797076 0.07739754 0.3412333 -0.07549025 -0.22301851 0.011751653 -0.17514169 -0.24184775 -0.35437453 0.16179754 -0.062408 -0.06844249 -0.16315302 -0.02793457 0.08337896 -0.3180476 0.10060193 -0.022071201 0.03387103 0.1137596 0.23074493 -0.104555994 -0.057770107 -0.05770761 -0.13033783 0.13717557 -0.05088083 -0.02387027 0.15989672 0.17121181 0.035453193 -0.24430706 0.11019246 -0.06623087 0.3287545 0.06304783 -0.07404035 -0.03168245 0.08804956 0.0062234458 -0.039035458 -0.012049761 -0.05966489 0.20264432 0.04391806 -0.2876821 -0.2276258 0.053296648 -0.1230089 0.05603393 -0.27262324 0.13113374 0.056734405; 0.018649636 0.3716111 0.08218696 -0.2963853 -0.34656137 0.05291685 0.15227109 0.09619568 0.16399497 0.18708466 0.1709432 -0.17064151 0.038787834 -0.091552906 -0.107152164 -0.17399101 0.16094455 0.022787325 -0.049017105 -0.025462225 0.2655594 -0.010532227 0.16849376 0.32589638 0.0499431 0.36147678 -0.038727112 0.092688814 0.04039149 -0.20452395 -0.11189697 -0.112253405 0.2776451 0.17967212 -0.2655871 0.32041943 -0.13029805 0.23607078 -0.30800322 -0.08155242 0.11172292 -0.0021293424 0.03297998 -0.18879098 0.20841603 -0.041502923 0.32068184 0.05573338 0.23877822 -0.17524391 0.36870074 -0.091109216 -0.001163067 0.14992033 0.1350963 0.045950055 -0.05364809 0.069102176 -0.038526915 0.052360486 0.0054325564 -0.06124449 -0.107365645 0.27590156 -0.15041314 -0.10658289 -0.0052207927 -0.32704145 0.20962436 -0.2498493 -0.059319098 -0.07504694 0.26843718 0.34997013 -0.117838964 0.105428055 -0.13068967 0.16755033 -0.12867293 -0.003533942; 0.24045835 -0.21066342 -0.15780205 0.10465387 0.008851583 0.030645894 0.05692166 0.106877044 0.098236285 0.033351313 0.12146406 0.056609716 -0.039469097 -0.092506036 0.023629162 0.21344526 0.046875417 -0.028173797 0.044595405 -0.31216943 0.20595099 -0.07238665 -0.078380466 0.03896557 0.10740785 -0.26815218 -0.13327897 -0.22323158 0.15841165 0.2743039 -0.15961382 0.0072893673 0.020634938 0.1411189 -0.1922467 0.015764322 -0.18470964 -0.20778415 -0.116824634 0.14160623 0.04739984 0.09604278 -0.23208697 -0.10582835 0.07815821 -0.29660553 0.07834879 0.06910638 -0.04262105 0.12023327 -0.22927211 -0.27756634 0.07638955 0.06848182 0.016356252 0.14279562 -0.17566559 0.08042993 -0.2169582 0.13537215 -0.14733186 -0.037994217 -0.08633833 -0.08695153 -0.049260274 0.0625499 0.16833843 0.17206505 -0.11251964 -0.0026264628 -0.0032556436 -0.18680973 -0.00083974795 -0.17916171 -0.023935428 0.008365131 0.08282621 -0.20445497 -0.00606386 0.09287928; 0.023138504 -0.2701387 0.033152692 -0.16511492 -0.016746521 -0.29233417 -0.07425634 0.08511931 0.013025857 0.08697235 0.14859016 -0.1645221 -0.13637438 0.023795972 -0.059198324 -0.03685286 -0.1491709 -0.06855824 -0.08282108 -0.08025936 0.11143422 0.16388 0.06352384 0.058216237 0.23631257 0.091563195 -0.13289036 0.088353865 0.05030541 0.024309337 -0.1646178 -0.05391214 -0.15108298 -0.060425356 -0.18102098 -0.12194091 0.18163274 0.0037794414 -0.12994951 0.18192297 0.2410465 0.022570686 -0.11545113 0.09974292 0.009566863 -0.1915923 -0.05056463 -0.019512542 -0.058785982 -0.073637016 -0.13891557 0.02004935 0.23986633 -0.0965314 -0.06101487 -0.17849499 0.016143085 0.22435653 -0.10049377 0.09193384 -0.07726551 0.20112333 -0.12574896 -0.23238966 -0.20489742 -0.041374084 -0.1018259 0.12672634 0.018494887 -0.017801605 -0.017549934 0.11879161 -0.097077735 -0.23842606 0.04945775 0.085026205 -0.1990796 -0.12512311 0.13860995 -0.069730066; -0.030830259 -0.042458456 -0.03988014 0.027545296 -0.16646522 0.12540433 -0.0491777 -0.013158883 -0.07786988 -0.09770396 -0.20450926 -0.102589265 -0.06421553 0.104265735 -0.023522874 0.013889766 -0.17516494 0.21104926 0.045350637 0.28888026 0.1258214 -0.032101247 0.05306835 0.15738037 -0.3203087 0.060710482 0.06481645 -0.04858212 -0.13449547 -0.14083017 -0.10243868 0.17224722 0.15226741 -0.17789361 0.03875524 0.20096356 0.15734613 -0.17727229 0.07871932 -0.023459943 0.12878174 -0.16167906 0.24955523 0.21950862 -0.09541185 0.29295573 -0.15735844 -0.12897801 0.31722546 -0.10774646 -0.026481409 -0.069587804 0.050900903 0.15313037 0.10461621 0.006093542 0.16252197 -0.11560453 -0.00013642566 -0.061167717 -0.2234791 -0.08556754 -0.06778854 0.08859275 0.12140016 -0.16003068 -0.14876464 -0.024556544 0.124098055 -0.078578085 -0.22545598 0.18139778 -0.03675867 0.12293552 0.14968087 -0.053600337 0.08871726 0.027219638 0.048215836 -0.04614216; -0.21374263 0.15563986 0.03905469 0.078272745 -0.2859685 0.28850827 0.1507147 -0.105993696 0.056572802 0.06644526 0.041234 0.06759703 -0.11597834 -0.11563822 -0.06600237 0.05928193 0.09346421 0.0323443 0.27095622 -0.010549322 -0.17469633 0.022235634 0.02325571 0.07242338 -0.22310212 0.17587115 0.13780254 0.2545951 0.11180621 -0.049884837 -0.06560941 -0.104230784 -0.0926695 -0.1395713 0.18649197 0.07272264 0.09509013 0.18111904 0.20162143 0.10316293 -0.09631264 0.19937368 0.16848345 0.047761574 -0.11637529 0.011150638 0.1935231 -0.06869273 0.21227182 -0.04260855 0.22044618 0.15935287 -0.060499284 -0.14372836 0.026585309 -0.17371917 -0.2127717 -0.02406485 0.1248263 -0.13275567 -0.12702784 -0.103606716 -0.0070274957 0.24250893 0.00965638 0.04898276 0.09835795 0.13907686 0.13203487 -0.20721881 -0.13107386 0.18630812 0.07902778 0.054086316 -0.17586777 0.090252966 0.120781206 -0.038321275 -0.12401946 -0.0748731; -0.10278562 0.23169845 0.28199863 -0.05478669 -0.21975474 0.2888645 0.15486528 -0.28918755 0.38537502 0.14737168 0.085345246 0.077082515 -0.037772827 0.04395744 -0.052038044 -0.05847307 -0.06337242 0.030092817 0.28339806 0.058126654 -0.24360487 -0.08892111 0.0035687597 -0.013365149 -0.10550944 0.059050523 0.12614462 0.14788322 -0.118155986 -0.4273529 0.34062633 -0.08000856 -0.012993492 -0.094432354 0.2525508 0.21836442 0.014478307 0.098689325 0.04766324 -0.06647607 0.099093735 -0.02670161 0.28124285 -0.01806076 -0.0044452976 0.22011913 -0.03579807 -0.096939765 0.20599191 -0.14457323 0.011224863 0.25470957 -0.2210831 -0.088172235 0.27837592 -0.1509142 -0.13730343 -0.11123741 0.30712312 -0.2518516 0.09567219 -0.28007248 -0.11361956 0.20700315 0.08641066 -0.045447677 -0.089979984 -0.03947898 0.15280259 0.015457349 -0.29328087 -0.17038079 0.25981763 0.100587904 -0.09348114 -0.025906695 0.28107756 0.00018251804 -0.19789378 -0.24259982; -0.042607725 -0.17215508 0.0055538923 0.17627633 0.06019586 -0.053185567 -0.30200163 -0.19258034 0.041808728 -0.24703991 -0.131499 0.1493332 0.23124538 0.030384284 0.00642738 0.19420747 0.0052534235 0.16576041 -0.16067576 0.04570723 -0.0975869 0.23878197 -0.101623684 -0.014391864 -0.042744815 -0.31949532 -0.092905484 -0.24014455 0.091652304 -0.0012114074 0.20390543 0.12731914 -0.030381422 0.0329091 0.04973557 -0.17666823 -0.007069286 -0.08913266 0.0075020874 0.1812341 -0.14757662 -0.0949825 0.038153775 0.15638122 -0.19829842 0.13074298 -0.28598776 -0.12299435 -0.11198344 0.19462283 -0.24313104 -0.033321604 0.107830904 -0.22470579 0.032577306 0.20274846 0.028976671 0.039708793 0.1689766 -0.17378771 -0.24190113 0.24480926 0.16152044 0.0013500843 0.17873533 -0.14962347 0.13305011 0.15483002 -0.09564108 -0.07611041 0.029520368 0.23255429 -0.016727848 -0.2695568 -0.027686821 -0.039646056 0.020996543 -0.12493551 0.09243772 0.28452367; -0.023401242 0.0710548 -0.007105601 -0.16713755 -0.3215268 -0.10617223 0.15666418 0.033552237 0.118064374 -0.020381302 0.039496418 0.05313393 -0.17566171 0.024133269 -0.22617362 -0.06862873 0.27231705 0.03133516 0.07504075 0.028508743 0.09040912 -0.17186844 0.16815254 0.11831303 0.020450829 0.04237081 -0.011948662 -0.025704294 -0.14224505 -0.102258064 0.092484 -0.01461848 0.16290471 -0.05621044 -0.0808207 0.36998942 -0.09675143 0.19387938 -0.2687873 -0.181318 0.19886017 -0.15474397 0.12526706 -0.02548357 0.0790111 0.21691407 0.13030803 0.14701639 0.26887688 -0.15685886 0.024288697 -0.067398906 0.049757823 -0.02579012 0.31629816 -0.15820462 -0.21839176 -0.08430927 -0.05417609 -0.06878498 0.091414176 -0.18616211 -0.27051392 0.117075756 0.049465217 -0.2538356 -0.25047624 -0.015631229 0.28077906 -0.035025835 0.09315172 -0.13158673 0.085474744 -0.002428285 -0.0404396 -0.05230156 -0.21940844 0.22927195 -0.043865394 -0.12239881; 0.01793634 0.22943327 0.0015287036 0.026145885 0.06660479 0.0038057552 0.011653775 -0.1754911 0.22912203 -0.13489345 -0.13571535 0.022895373 -0.16590771 0.056811832 -0.004933934 0.017503895 -0.09940909 -0.06167259 0.07824349 0.042988356 0.025224118 0.07827075 0.10318968 0.15329921 -0.021169012 0.20760404 0.016303532 0.19506405 0.0793649 -0.21495083 0.21941575 -0.026899926 0.19505733 0.053009246 -0.071562074 0.034093313 -0.0066802255 0.0818099 0.0010122509 -0.060482215 -0.2007589 0.1852506 0.18736874 0.08774592 0.15163006 0.18784164 0.076358534 -0.24115418 -0.14146174 -0.018267192 0.12135918 -0.08066767 -0.22101738 -0.17145383 0.10292597 -0.1440018 0.13499984 0.11194596 0.046939317 0.023537092 0.09394944 -0.0850743 -0.022482641 0.17628735 -0.023343747 0.11472688 0.086135045 -0.15330663 -0.030782187 0.16889709 -0.043856755 0.0851224 0.23929718 0.18944733 -0.08137241 -0.16152424 0.20547149 0.049233045 0.11735348 -0.08343693; -0.20357671 0.021572867 -0.16749647 0.22195302 -0.2476693 0.29235423 -0.03490133 0.00031774637 0.055876464 0.12691103 -0.016135477 0.17115328 -0.16297688 0.044808008 0.13584004 -0.26042497 -0.121474355 0.20387687 -0.0025403579 0.00042853595 0.06174055 0.13420627 -0.26254064 0.061400205 0.0030564123 -0.057381626 0.11425101 0.09827601 -0.07416346 -0.13784473 0.09956476 0.18631811 0.110886075 0.10654674 0.033780843 0.2859555 0.062466126 -0.09293617 -0.10704469 0.04236556 0.056803767 0.07979852 0.09588942 -0.053818516 0.012184201 -0.07294144 0.085229345 -0.085625984 0.20989764 -0.07822563 0.2663051 0.01734492 -0.27676618 0.017431581 0.110623635 0.0039372607 0.04620531 0.0656656 -0.09893365 -0.19583173 -0.0062255994 0.01769433 0.17992505 2.2134278f-5 0.047414497 0.027306478 -0.0055012926 -0.13433027 0.16666506 -0.0064180344 -0.0020372174 -0.08527777 0.1592225 0.09971053 -0.1267231 0.06219162 0.17937116 0.14615475 -0.04383132 -0.21222512; 0.04658085 -0.2762974 -0.16732903 -0.00942134 0.09371405 -0.0918276 -0.12693986 0.14689377 -0.16271308 -0.23131737 -0.032269232 0.0041248 0.012106719 -0.03246878 0.049600597 0.049218774 -0.19087923 0.0330477 -0.0651518 -0.05772778 0.08822547 0.13467668 0.15268104 0.09103603 0.21195813 0.00387698 -0.023653926 -0.10223447 -0.06031657 0.3142201 0.009442529 -0.08658134 -0.08531568 0.042356327 -0.04821218 -0.17668712 0.060161524 -0.021734457 -0.09705483 -0.22474174 0.0973699 -0.21421815 -0.108861946 -0.1734455 -0.08771314 -0.05836039 -0.08772381 -0.048680626 -0.065668166 0.0600347 -0.24429072 -0.1918536 -0.007933409 0.021074515 0.02007669 0.029621042 0.1688703 0.11163106 -0.17060897 -0.07229752 0.0076752184 0.20403467 0.06164779 -0.15821499 -0.031191126 0.18431337 -0.065538175 0.14625564 0.008882635 0.19268152 0.2482203 0.1558838 -0.25594148 -0.1941591 0.14508478 -0.002103584 -0.24383238 -0.32878762 0.17137271 0.10379682; 0.0119931875 0.037156314 -0.10291498 0.1329489 0.05007362 -0.10428188 0.033536784 0.16036698 -0.081732474 -0.22192922 -0.047019176 0.06423836 -0.12882613 -0.18824086 0.09127727 0.13530406 -0.17766784 -0.23725514 0.0013530189 -0.14545123 0.043942317 -0.20599046 0.19898084 -0.2260812 0.02716092 0.03850501 0.04341816 -0.21185745 -0.04862546 0.1906036 0.06777349 -0.109760694 -0.038578864 0.15340093 -0.13594191 -0.022247307 -0.06836379 -0.053706463 0.122147866 -0.007979984 -0.067281336 0.066932485 -0.15279803 0.0846044 -0.03061984 -0.04868218 0.1040325 0.27683133 -0.0014139425 -0.05816166 0.043377236 -0.1918072 0.33739516 0.22828887 -0.15467821 -0.09502202 0.21381943 0.057206497 0.058095887 0.017553696 -0.051136956 0.07694388 -0.22055079 -0.05283227 -0.17529255 0.029879116 0.06763474 0.111666046 -0.10601157 0.08640359 0.33502147 0.022655014 -0.22841814 -0.22941433 0.11280584 -0.18428044 -0.2375 -0.19931526 -0.06343899 -0.0866912; 0.3124345 -0.2928221 -0.17160459 0.02074537 0.35173464 -0.21891813 -0.21913819 -0.051040236 -0.05688573 0.11848032 0.016729014 0.0048835883 -0.14997016 -0.09001769 -0.15346417 -0.02157867 0.10803142 -0.20315889 0.04801485 -0.068136744 -0.09082943 0.08115608 -0.04961931 -0.17140783 0.28876027 -0.1035949 0.1086656 -0.1705018 0.14206292 0.1499427 -0.24935825 -0.0068438384 -0.20092773 -0.18759196 0.05853148 -0.35407907 0.14076202 0.034156956 -0.024087284 -0.0050935075 0.109699555 0.13012058 -0.27009258 0.11294848 0.15695357 -0.13120134 0.0779461 0.20842041 -0.35212386 -0.066682905 -0.12704998 -0.15309046 0.0044410178 0.040452797 -0.10680621 0.080686904 0.23698731 -0.1375659 -0.17802145 0.25763404 0.2137553 -0.027445056 0.12276311 -0.29515123 -0.2008228 0.047835343 0.17182752 -0.050666504 -0.15647523 0.1853398 0.11744602 -0.052714918 -0.26528716 -0.073227756 -0.08307581 0.17478809 -0.11581048 -0.20035037 0.17851533 -0.033313256; 0.009703492 -0.17142116 -0.054418206 -0.14618818 0.28603992 -0.18394697 -0.23439632 -0.10136534 0.0511375 -0.009586938 0.06461649 -0.1479816 -0.003896756 -0.11665958 -0.19535416 0.08330511 0.09367286 -0.17468409 -0.1563288 -0.14487712 -0.11944691 0.060117505 0.09156159 0.0947531 0.14024147 -0.1666482 0.08834541 -0.17379425 0.16502309 0.14350149 -0.24837269 0.063708365 -0.23122862 0.034270123 -0.18031117 -0.11221088 0.10842016 -0.083872445 -0.005556264 -0.06936452 -0.042897537 -0.021712478 -0.26620784 0.124072134 0.14110042 -0.15125006 0.0029815938 0.032870203 -0.22184817 -0.2130826 -0.102666505 -0.12039925 0.031405374 0.045106947 0.007641403 0.16922985 -0.12005715 -0.21901712 -0.07729146 -0.09613293 0.017244868 0.033643417 -0.18499169 -0.05561657 -0.06295395 0.005433262 -0.14102925 0.13390572 -0.18271255 0.30003497 0.17674254 -0.13017915 -0.0045397477 -0.11764011 0.18575656 0.009599317 -0.0906294 -0.035701975 -0.019939674 -0.061055295; 0.0056063244 -0.27409497 -0.020041935 -0.10920912 0.14067616 -0.02793863 -0.12840576 -0.03322768 -0.32693562 0.107741654 0.07238184 -0.1689599 -0.07169354 0.10444052 -0.2118166 -0.02183387 -0.051616274 -0.12836736 -0.02673517 -0.028015634 0.09136759 0.15980858 0.08464197 -0.0710885 0.26352087 -0.11400056 -0.08953314 0.1037674 -0.092994764 0.09680416 -0.16438417 -0.2084551 -0.22402233 0.19238573 0.11689125 -0.16436361 0.17346719 -0.08207514 0.07333485 0.102543615 0.08549032 -0.10438473 -0.1957353 -0.16003689 -0.056616206 0.023882568 0.17053361 0.26474783 -0.22702686 -0.069593824 -0.2287018 0.10152146 0.09440744 0.06000479 0.102203965 0.058524244 -0.027032623 -0.15817651 -0.17542124 -0.10045998 -0.0655576 -0.039323505 -0.06551265 -0.042136155 -0.16767141 -0.14202943 0.15977345 0.08184235 0.07409968 0.24829066 0.18330188 0.16910379 -0.19768378 -0.09153396 -0.1506827 0.041500993 0.069360584 -0.055633735 -0.052850388 0.07571037; -0.29517072 0.16226746 0.23526484 -0.049992416 -0.15391643 0.12690176 0.28896162 0.006163032 0.27364188 -0.002193702 0.16696467 -0.04025119 -0.08847961 -0.077916466 -0.03378502 -0.020816762 -0.056594115 -0.09673719 -0.018483212 0.18928143 -0.1795831 0.072306745 -0.19125755 0.17219552 -0.21977024 -0.003601824 -0.06867711 -0.11981025 0.011361857 0.013364223 0.024957895 0.21089469 0.15888272 -0.18292634 0.10545734 0.12529573 -0.01734593 0.024337681 -0.1196675 -0.13288483 0.05275006 0.12202002 0.22045258 0.0925308 -0.08146622 0.21609667 0.05592966 -0.04768451 0.20717251 0.19518799 0.13021095 0.0936235 -0.3119842 -0.11993237 0.20428017 0.032458376 -0.17057085 -0.20151821 -0.010736057 0.0012857893 -0.1640322 0.10422023 -0.043363374 0.01472387 0.099525444 -0.17565143 0.14262663 0.12882087 0.128873 -0.31057945 -0.13665004 -0.10486974 0.3200016 0.109249905 -0.19606018 -0.12153422 0.08400006 -0.02482058 -0.10285934 -0.0036161148; 0.23228025 -0.00030634785 -0.122923546 0.027765526 0.27671593 -0.054696687 -0.05638993 -0.0010526401 -0.16304465 0.17576781 -0.120989 -0.030014403 -0.01379725 -0.08445724 -0.2407784 0.22694016 -0.029567182 -0.1256474 -0.08989162 -0.13884346 0.021733547 0.15513152 -0.04690073 -0.06992783 0.013833145 -0.010336887 -0.19086923 -0.20599528 -0.013418927 -0.055654455 0.014092624 -0.110926375 0.13557754 -0.065175004 -0.16890526 -0.10746333 -0.21030173 -0.1797299 -0.027470127 0.089681745 0.14731131 -0.1160537 0.008406481 -0.09523098 0.08561357 -0.16561535 -0.034947168 0.09148117 0.012960111 0.061668728 -0.07352976 -0.22020932 -0.029519888 0.1487526 0.081527136 -0.08965164 0.061020624 -0.041330628 0.13828093 -0.0044895355 0.13638592 0.14826652 -0.22666775 -0.09073005 0.10678477 0.036332257 0.1792224 -0.1625522 -0.06298376 -0.0665483 0.28343034 -0.08457425 -0.18071723 0.11986406 -0.105811045 -0.11874387 -0.04617386 -0.28853253 0.12775764 0.08583523; 0.05138477 0.105758145 0.10637038 0.04370692 -0.05785189 -0.038830522 0.23623462 0.03460333 -0.03953753 0.17826964 -0.069381125 0.046254262 -0.06252537 -0.11467984 0.16916217 -0.08049476 0.0084337955 0.09684797 0.30416793 -0.064259276 0.005685651 0.07519367 0.042439952 -0.18983386 -0.27873453 -0.053207584 -0.1416942 -0.007570604 -0.08665119 -0.20918986 0.1627425 -0.08189044 0.100935265 0.08312694 0.21056214 0.1559585 -0.07672645 -0.040961504 0.19846244 0.17739607 -0.21689613 -0.21255893 0.090307645 -0.020379815 0.016533203 0.25153127 -0.021279221 0.071531706 0.3091682 0.12535024 0.102241896 0.20848265 -0.05574515 -0.07438387 -0.024201611 -0.0154871335 -0.11697979 -0.10432702 0.107196376 0.11993899 -0.023673173 -0.009366486 -0.14164884 0.007074111 -0.02903721 -0.034868043 -0.054961607 0.22261544 0.02201677 -0.07719352 -0.0005722003 0.16463272 -0.08426901 -0.03684434 0.035944313 0.0013987436 0.0718002 0.08681551 0.013051334 -0.1041897; 0.16232009 -0.25551614 -0.11273689 0.22684947 0.2651646 0.015742933 -0.23917024 -0.058883008 -0.21111153 0.009250984 -0.2955283 -0.1525288 0.07825973 0.071709186 0.07108012 0.20253065 -0.22906438 0.112143844 0.09216023 -0.03777494 0.054652732 0.20150326 0.040040847 -0.29800203 0.19261386 -0.24067435 0.060747173 -0.06344173 0.22143312 0.27897978 -0.07078779 -0.12345294 -0.3917355 -0.29791486 0.048377994 -0.17696957 0.2356726 -0.17820498 -0.061010405 0.07899335 -0.043136403 -0.014312027 -0.14103113 0.2863253 0.02703639 -0.16186562 -0.13389233 -0.15775941 -0.026663793 0.047036618 -0.069053225 -0.088645436 0.24868889 -0.0019536817 -0.26749262 0.17177315 0.10299323 0.022679137 0.029326813 -0.18896545 -0.02500056 0.13100144 0.11944312 -0.035177153 -0.05524776 0.032405693 0.07638141 -0.042568054 -0.24862154 0.060189452 0.26819915 0.123946585 -0.31409934 -0.09994446 0.26390076 -0.10828489 0.0755048 -0.3558088 0.14965472 0.007641987; -0.22015372 0.3232947 0.16009049 -0.12377076 -0.10005308 0.14087614 0.09691699 -0.029869162 0.2255759 0.19630101 -0.13463748 0.2247205 -0.013097234 -0.055622928 0.26072264 0.18898576 -0.05026999 0.20370644 0.066927575 0.20409638 -0.1738717 -0.1211255 -0.0890241 -0.11161788 -0.25508344 0.3137829 0.21359617 -0.00080908684 0.019392306 -0.3643864 0.050328277 0.2834537 0.19041929 0.13081405 -0.069820434 0.13138139 0.09343413 -0.10394661 -0.01161809 0.10710102 0.10029187 0.024298312 0.13412432 0.002081239 0.017585315 0.17696588 0.12621018 -0.26300216 -0.0050758678 -0.023132773 0.19871205 -0.16253206 0.017625093 0.062002577 0.065704085 0.04283549 -0.12801778 -0.07451887 0.23523882 -0.22755268 0.115291506 0.059244804 0.18052645 0.206431 0.07350881 0.0070757065 -0.20196503 -0.110277526 -0.03802076 -0.20912898 -0.25572085 -0.05040105 0.24312212 0.13514401 -0.14328714 -0.037201524 -0.014166217 0.031320244 0.1523666 0.032577172; -0.04361893 0.020124516 0.16724308 -0.06288588 -0.17726816 -0.11580894 0.41316536 -0.07205287 -0.072879076 0.19899815 -0.030381912 0.02126943 -0.081327975 -0.14556348 -0.0020577442 0.0138987955 -0.084578335 -0.021775411 -0.001064375 -0.17993513 0.21422976 0.09068058 0.061242938 0.27284715 -0.08065877 0.29316208 0.098667495 0.054405477 -0.20530406 0.038985014 0.08656785 -0.13216399 0.13225861 0.15645087 -0.14294821 0.26408553 0.10435552 0.08971988 -0.14574021 -0.0073607746 0.004535554 -0.18895827 0.2230256 -0.18150043 0.1649315 -0.1473168 0.18588959 -0.0804223 0.29833367 0.093409695 0.2441775 0.12306806 0.056839578 0.022686087 0.18496913 0.09977674 0.102157384 -0.1410639 -0.20027797 0.2567232 0.12143822 0.040726643 -0.00455464 0.24660763 -0.009310745 -0.06535713 -0.23403397 -0.25657246 -0.040328275 0.07068373 0.16273515 -0.061179463 -0.10100395 0.22438993 -0.19746508 0.14875641 0.13741472 0.18156564 -0.13193873 0.05862836; 0.27714512 -0.119051576 0.03418637 0.099700905 0.357411 -0.04347574 -0.15357211 0.09293949 -0.24965522 0.015667528 -0.13069001 -0.01991703 0.006932301 -0.07066508 -0.20129064 -0.09382758 -0.040774174 0.10696892 -0.10837111 -0.10222906 -0.15753965 0.06480378 -0.23236421 -0.19979852 0.16458823 -0.34932858 0.23110609 -0.32899493 0.0040657595 0.18314262 -0.07592211 -0.12088312 -0.14433955 -0.045098282 -0.02418567 -0.07165491 0.13924657 -0.081075534 -0.1331085 -0.09806463 -0.16780376 -0.11255453 -0.28650826 0.16185437 -0.040648486 -0.14717858 -0.052301746 0.030937197 -0.28808424 -0.022323303 -0.049158458 -0.14725587 -0.038550917 0.12757933 -0.11150923 0.22547749 0.26779512 -0.08748538 -0.14831625 0.13084415 0.09782095 0.05942883 0.094623506 -0.28194773 0.057769034 0.093076445 -0.054109666 0.115761325 -0.017528363 -0.053781446 0.30933708 0.09330125 -0.18519858 -0.07663559 0.13125409 0.091198005 -0.16424212 -0.007860903 -0.07615673 0.032541532; 0.05916078 0.15244547 0.1153206 -0.16521482 -0.26320648 0.100138575 0.21351528 0.013113929 0.14324121 -0.089296564 -0.042936802 0.01228004 0.046172135 0.03884529 -0.07379583 -0.14458165 0.01456651 0.21308695 0.16037603 -0.061343934 -0.028720776 0.094537854 -0.012508935 0.063720524 0.06456895 -0.07212796 0.022393249 0.15309976 -0.01885543 -0.12681375 0.12896349 0.16717266 -0.010611385 -0.07352756 0.034699846 0.16794601 0.09228669 -0.12480523 0.19316125 -0.022072714 -0.17745808 0.05376403 0.22337662 -0.17186338 0.13645358 0.08378934 0.019893939 -0.05138705 -0.0038667002 0.084206015 0.20562743 0.049533453 -0.19620381 -0.18328059 -0.07643283 0.0849285 0.13726062 -0.035891395 -0.06191759 0.21307378 -0.12969735 -0.14354414 -0.040017728 0.035057575 0.13113834 -0.02436862 0.10813104 0.180507 0.02551646 -0.054489996 -0.030640265 -0.048093405 0.119150735 0.18276022 0.098119184 0.0021400754 0.07882104 0.19362496 -0.15746522 0.13980974; 0.046570815 -0.28676015 -0.059300564 0.035385173 0.14962365 -0.24606378 -0.085438594 0.019038409 -0.047012668 -0.051499184 -0.013396974 -0.17470814 0.1428314 -0.2072189 -0.010115829 0.139315 -0.14127268 0.06679733 -0.23756638 -0.23201291 -0.06461889 0.019493354 -0.10686758 -0.02922122 0.1729826 0.10414427 0.102007 0.120714314 0.080460936 0.087615326 0.027616959 -0.08681866 0.05247344 0.11720558 0.11336467 -0.16552617 0.090687655 0.060169827 -0.12539984 0.076002255 0.17372 -0.08715454 -0.26332253 0.12926781 0.12882064 -0.19256482 -0.021135833 0.26973277 -0.13455126 0.14025429 0.07906557 0.05308636 0.14418514 -0.1519399 -0.008540427 -0.07736129 0.19878131 0.06654681 0.04926872 0.0017104393 0.08829734 0.11384762 0.081200875 -0.25362965 0.07415466 -0.12738323 0.17772633 -0.11399912 -0.2230828 0.104017936 0.071117006 -0.036217034 -0.15453213 -0.14324318 0.14425267 0.11014429 0.020656046 0.01927609 -0.12816885 -0.06431219; -0.010777804 -0.13109666 -0.131462 -0.020706695 0.001292609 -0.25648904 -0.25175065 0.2256988 -0.17513901 0.0742723 0.07844644 -0.30505013 -0.057159673 -0.06394415 0.07452251 0.012135869 -0.0390588 -0.0833278 -0.016358856 0.04037104 0.08981074 -0.13880256 0.061829112 -0.021167325 0.22493184 -0.16587183 0.23757204 0.067548424 0.04573547 0.14956063 -0.30222693 -0.06929315 -0.23154171 -0.2353356 0.092902735 -0.051245626 -0.13745712 -0.27297726 0.20490846 -0.11510901 0.090520985 0.14299834 -0.26978293 0.15275523 -0.014469691 -0.005962898 -0.008062115 0.024343971 -0.23799098 0.24360016 0.0018722331 0.057009693 -0.020174822 0.05248765 -0.22981161 -0.014006291 0.07521682 0.07469221 -0.105574936 0.077223174 0.14653796 0.15213974 -0.16611046 -0.24799708 -0.08016666 -0.002082208 -0.089174345 0.18916261 -0.07087692 0.24179061 -0.04228902 0.10570391 -0.31207862 -0.17813945 0.06707517 -0.1618765 -0.10789696 0.065187834 0.22635408 0.0683238; 0.04528663 -0.33748624 -0.0597115 0.03089765 0.29634362 -0.350539 -0.11418119 0.31310466 -0.4087335 -0.027250223 -0.14993398 -0.19259085 0.122778215 -0.06854091 0.024049312 0.08639527 -0.23964486 -0.13090764 -0.1788316 -0.2564859 0.0028486182 -0.023926862 -0.051642414 -0.28667685 0.22975291 -0.089542024 0.19674361 -0.12680002 0.075961605 0.15077205 -0.32331204 0.064320505 -0.07178338 -0.26793745 -0.13599496 -0.37644422 0.06088847 0.018190527 -0.030930975 -0.1209877 -0.06059224 0.06317284 -0.40645292 0.11064075 -0.22010279 -0.06460872 0.055432156 0.1872339 -0.17094678 0.12232035 -0.20272493 -0.04311285 0.20431288 0.25662458 -0.10743339 0.09973653 0.3284179 -0.05375385 0.037685364 0.27383953 0.020653449 0.20428051 0.0770995 -0.3694073 0.092165835 0.258231 0.018829187 0.027085908 -0.071159765 0.15707994 0.28613922 0.00545187 -0.28477862 -0.32501966 0.053957008 0.035587702 -0.22663166 -0.31963986 0.0062968284 0.06187599; -0.028295878 0.13517134 0.22729576 0.14192188 -0.34580496 0.15119198 0.2764298 0.10169851 0.3945726 0.10881995 0.05927177 0.22337797 0.07700503 0.04599869 -0.086161435 -0.3390187 -0.042644337 0.043538123 0.29471418 0.15670803 0.11744038 -0.097821034 -0.09353644 0.15760076 -0.13297813 0.30354485 -0.07123432 -0.011175034 0.15396321 -0.2897684 0.043701634 0.20020336 0.020747967 0.20356306 0.22848043 0.24595676 -0.1862193 0.033645585 0.12261309 0.18696553 -0.21114092 -0.016257823 0.42552224 0.19616337 -0.08824448 0.08736191 0.2262015 -0.11003906 0.34670115 0.005328011 0.295324 0.033327512 -0.11508837 0.06156732 -0.04055329 -0.18781906 -0.27482358 -0.2056138 0.28117633 0.07047053 -0.24671729 0.013565114 0.11767868 0.07219152 0.12015222 -0.053410985 -0.1485745 -0.17032343 0.20896935 -0.14477645 -0.052369494 0.13411029 0.017994365 0.37291473 0.07247564 0.22333625 0.02581669 0.14730226 -0.19456294 -0.27822018; -0.22703883 0.07597353 0.05130245 0.020725364 -0.09650247 0.32010135 0.3039942 0.05793721 0.39006588 0.21671823 0.14935327 0.25589705 -0.19144714 0.01448421 0.1872988 -0.0118605355 0.06735292 -0.110790655 0.35984704 -0.03654206 -0.052007988 -0.07184914 0.12864734 0.09335482 -0.378242 0.12928785 -0.08204114 0.15830569 -0.08941993 -0.0872078 0.12340121 0.15264577 0.08990992 0.0963087 -0.19098087 0.21659546 -0.11163285 0.11405455 -0.22649696 -0.24993588 0.17466773 0.01648206 0.3045745 -0.102064535 0.063280255 0.25896844 0.2339056 -0.31908405 0.11033319 -0.2735012 0.41267902 -0.061327808 -0.33374482 0.10332717 0.22350729 -0.25663283 -0.1535102 -0.22590345 -0.040078644 -0.051756863 -0.016240118 -0.19003937 -0.119701594 0.27163428 0.007994034 -0.046157483 -0.34686345 -0.12085909 0.16052002 -0.21541639 -0.3985371 -0.1643341 0.027810648 0.19446577 -0.013623889 0.16295698 0.20186089 0.10322163 0.034398556 -0.23487991; 0.104578026 -0.11322345 -0.106759004 -0.15741545 0.15007268 0.07196019 -0.13233869 -0.040201087 -0.07616178 -0.046079237 0.03888243 -0.13480698 0.06949956 0.023818964 0.01644538 0.27989495 -0.11990993 0.05326505 -0.017403703 0.04785792 0.15058234 -0.08666181 -0.020645149 -0.13414173 0.23435797 -0.22008742 0.044061095 0.038494065 -0.16448957 0.06881672 -0.18937235 -0.15075907 -0.07857702 -0.11650191 -0.035671294 0.06911305 0.016605215 -0.062122364 -0.1221939 -0.020289289 -0.14687324 -0.036249798 -0.3254114 0.023443215 0.22679994 0.10257589 0.1848515 0.29988164 0.006948581 0.09913636 0.041587707 0.036174797 -0.025477676 -0.09349941 0.15856315 0.014119645 -0.1580039 0.030359099 0.07092574 0.09047479 -0.056820825 0.026985839 -0.21631232 -0.18965773 -0.15890864 -0.15605626 -0.065664366 -0.08767716 -0.16300498 0.23381725 0.1915573 -0.16493379 0.013317065 -0.08057401 0.071174815 0.07424434 0.07711205 -0.257369 0.16687042 0.16754633; 0.20850386 -0.24163967 -0.08062233 -0.12858894 0.06973426 -0.24111322 -0.21263127 -0.09209939 -0.22151475 0.13691038 -0.12142818 -0.21865748 0.090904735 0.0560778 -0.02153284 0.11518431 0.16492957 -0.14886439 -0.21471307 -0.27581048 -0.018095367 0.045937333 0.25336695 -0.18947344 -0.096572444 0.11687119 0.0022277154 0.17950454 -0.06920948 0.06561514 0.13935791 0.1024656 -0.11707066 -0.017078537 0.11279405 -0.20269872 0.15022577 -0.03037782 -0.1431616 -0.1526368 -0.031506315 0.027085945 -0.19503726 -0.058414366 0.22942226 -0.18468896 -0.001600395 0.027268762 -0.2872541 -0.17357105 0.11210046 0.12419432 0.1986403 -0.054602828 -0.13488036 -0.011539474 0.15366092 -0.022988852 0.17259587 0.038203128 0.11309085 -0.16246448 0.105136015 -0.30290166 -0.10861988 0.13302591 -0.132418 0.05276103 0.0013626604 0.15761304 0.030943846 0.081717834 -0.102262415 -0.20862404 -0.10162057 -0.08021961 -0.1522397 0.07947214 -0.036686834 -0.09773686; 0.24306104 -0.19874267 0.050599594 -0.14192027 0.029911064 -0.31274495 -0.2897393 -0.09200033 -0.27469307 0.033122946 0.021301558 -0.055167392 -0.11160263 0.13117242 -0.25250652 0.21353209 0.149781 0.038247235 -0.0017794564 -0.25389904 -0.07102401 -0.0663304 -0.059306294 0.11960223 0.19736199 -0.013157256 -0.046863556 0.07113346 0.12849596 0.2349974 -0.057036623 -0.12960832 0.033393543 -0.1268364 0.054973982 -0.30767405 -0.12284681 -0.012031762 -0.16091917 -0.10555244 0.08576316 -0.1266924 -0.14430568 -0.030190865 -0.16060823 -0.13213506 -0.04357611 0.03459827 0.00022833029 -0.13380383 -0.07761903 0.0022797568 0.038680404 -0.13239971 -0.10234868 -0.08791346 0.03831778 0.09469367 0.049677152 0.086199 -0.05353432 -0.050228618 -0.18237329 -0.0013193995 -0.08922476 0.031004822 -0.03436615 0.0075644585 0.009660691 0.054374196 0.24595769 0.036151923 -0.058734596 -0.085358664 -0.15768184 -0.008291409 0.037045028 -0.18006088 -0.0659948 0.052885585; -0.13823038 0.20799899 -0.24513525 -0.14877503 -0.21741547 -0.028713664 0.0065010544 0.10404322 0.028159205 0.13251565 -0.051800504 0.08972752 0.22261378 -0.14624181 -0.12777942 0.06758822 0.060196456 -0.020681977 0.19540979 -0.023078842 -0.21051455 0.044985663 -0.18605916 0.103023104 -0.23284474 0.15316296 0.13274197 -0.13932858 -0.12715021 -0.15139322 -0.11502476 0.095336795 -0.17398575 -0.14725132 0.06448194 0.23946732 0.09766724 0.13531916 -0.11743754 -0.0033478495 -0.2056134 -0.14763927 0.21752493 0.008389881 -0.035941005 0.22986475 0.1176851 -0.09463026 0.093030654 -0.110110916 0.017299382 0.0040558465 -0.17547354 -0.1241284 0.12866642 0.19622444 -0.09678339 0.116018265 0.17297132 0.044692967 -0.107364304 -0.080439426 0.24444626 0.03422326 0.086436845 0.005855868 -0.11944649 -0.039612852 -0.037046537 -0.14584273 -0.09462752 -0.028433736 0.19629927 0.21816671 0.15325725 0.006753303 -0.13957882 0.05525119 0.06487579 -0.08676317; 0.07730091 -0.06625814 0.1909905 0.089306414 0.09269011 -0.08556469 -0.24749057 -0.030280447 -0.22515623 -0.073760964 0.060001966 -0.069526166 -0.12009836 -0.14867283 -0.15072058 0.19721821 -0.100271024 -0.19322892 -0.13311823 0.019453393 0.017715964 -0.06557426 -0.00020121183 -0.07735869 0.18212059 -0.072365135 0.14375308 0.17783634 -0.13995951 -0.07498661 -0.19359954 -0.17221555 -0.031821087 -0.065087266 -0.14548667 -0.043262426 0.08459518 0.17857917 0.137677 -0.07619632 -0.029508395 -0.03015848 -0.22931828 -0.038426682 -0.16515686 -0.009139707 0.076069675 -0.018783526 0.026661575 0.04288777 -0.140906 -0.17823577 -0.0006467492 -0.15153627 -0.020230716 -0.08237843 -0.047232028 -0.08387652 -0.015479172 0.16132668 0.07014649 -0.15486059 -0.17118937 -0.30011103 -0.05328926 0.0030985463 -0.09543371 0.01772397 0.0888159 0.05630563 0.30299675 -0.124789305 0.05268292 -0.020192511 0.038570315 0.093343675 -0.11107488 -0.21339461 -0.04888779 0.1868051; 0.28424153 -0.26158464 0.074174955 0.07090519 0.13839425 -0.015933024 -0.13473968 -0.06675588 -0.1965897 -0.18181744 0.19959109 0.012169373 0.103688315 -0.056551754 0.028338972 -0.090660185 -0.03328066 0.08235943 -0.110186785 -0.06573726 -0.10919681 -0.024678508 -0.016033739 -0.015291712 0.057362758 -0.31964406 0.14763951 -0.24844879 0.096640326 0.29481706 -0.21565701 -0.028167777 -0.09918527 -0.0821643 0.062590405 -0.2941626 -0.12191127 -0.038314886 -0.18200383 -0.14134867 -0.10685778 0.10950489 -0.26896003 -0.11381029 0.031850718 -0.21141793 -0.20402656 0.12277226 -0.122986905 0.15428175 -0.17545451 -0.19734788 0.11024039 0.009963282 -0.2163936 -0.12698735 -0.040045135 -0.03749312 0.0099195745 0.118871875 0.08951695 0.28598055 -0.17379871 -0.1337978 -0.09382576 -0.009439326 -0.116518535 -0.0023895516 0.02766318 0.07169353 0.068965286 0.02927159 -0.3564367 -0.102486916 0.052305914 0.025661996 -0.026812544 -0.10571813 0.28102863 0.044283945; -0.091297 -0.32847703 0.18552366 0.08547231 0.30184153 -0.05384612 -0.011550133 -0.05523977 -0.2560976 0.06784958 -0.022339333 0.034767836 0.12969024 -0.030168325 0.036859225 0.029954365 0.121742755 -0.23358576 -0.14374818 -0.043484356 0.16009611 0.06453795 0.22135942 0.12283906 0.04116474 0.026972322 0.008769466 -0.0513567 -0.08084259 -0.05815388 -0.10422668 0.035998024 0.089127265 -0.07724047 0.100857474 -0.24808119 0.15078707 0.06904276 0.024379246 0.075195596 0.05711406 -0.024176333 0.003536471 -0.019507783 0.17760491 -0.31031114 0.04867877 0.009930088 -0.04682667 0.099927954 -0.07265345 -0.18070108 0.051582422 -0.12489401 0.049506545 0.047604535 0.025259161 0.17173211 -0.2116942 -0.025523722 0.17804815 -0.12680219 -0.043376677 -0.0064355694 -0.1341542 0.026399797 -0.03907655 0.06247885 -0.25469747 -0.020408873 0.18018803 -0.03978384 -0.21354963 0.071997024 -0.0127052115 -0.13561375 0.11937562 -0.21272981 0.15855457 0.06663535; 0.21694103 -0.035628416 -0.10098991 -0.068467006 0.008227788 0.052764334 0.01384813 0.21107203 0.051923834 -0.14127788 -0.019383991 0.045310598 0.16754921 -0.083599314 0.03699555 0.20154683 0.059498854 0.15505588 0.08097419 -0.057207905 -0.032443482 -0.014716078 -0.13672753 0.15336706 0.17359836 -0.09992813 0.13176844 0.033588644 -0.035142012 -0.001699067 -0.0702557 0.14557429 0.07016535 -0.106171004 -0.16882609 0.08433696 0.17655663 0.11116146 0.00613504 -0.10081269 0.16610208 0.19089665 0.042507622 -0.19627665 -0.10137817 -0.16330948 -0.019912887 -0.040279545 0.0017189602 0.13182174 -0.06910916 -0.09391647 0.30141076 0.08187713 0.17738025 0.08329931 0.15125722 -0.12643121 0.11372891 -0.011480116 -0.07275165 0.15401153 -0.037827604 -0.10113573 -0.08866325 0.07171726 0.1841609 -0.07091389 0.0051687704 -0.026531044 0.068955906 -0.053816322 -0.24489255 -0.18183124 0.08157656 -0.068424 0.061748452 -0.16160963 0.20038188 0.014033116; 0.07272677 0.031102303 -0.05360744 -0.20526424 0.3101187 -0.17232756 -0.17209828 -0.11182025 0.048776805 0.0460523 0.18350238 -0.026874207 -0.06570437 -0.15138099 -0.13002923 0.19158848 0.04254302 0.038963348 -0.33337662 0.025126131 -0.111421764 0.06309912 0.15199853 0.011656923 0.06497597 -0.14496867 0.096034944 0.00041919667 0.17727433 -0.0689608 0.043418758 -0.14582802 -0.09391967 0.16086112 0.08404687 -0.2900188 0.008609965 -0.10558559 -0.20966136 -0.112243876 0.10298276 -0.106608145 -0.127579 0.14290598 0.15618254 0.045296654 0.206664 0.07530468 -0.1765359 0.0599346 -0.12534839 -0.22034252 0.06443666 0.19344963 -0.11816932 0.016188119 0.033035435 0.00942613 -0.123890825 0.10549554 0.023499675 0.21684337 -0.22008853 -0.026315289 0.13882424 -0.06172469 0.17196423 0.09870589 0.036604837 0.22440287 0.23740387 0.13713346 -0.28620002 -0.275408 -0.20735748 0.13835618 0.00043388415 -0.07281233 -0.053202104 0.13764413; 0.23080553 -0.027020514 -0.17230256 0.15105405 0.06467943 -0.031859998 -0.20659037 -0.03513129 -0.325373 -0.029375896 0.04677056 -0.1967998 -0.10725149 0.08296699 -0.22694796 0.13999288 -0.070552394 -0.11999038 -0.030555025 0.07288794 0.16915391 -0.09707127 -0.056845624 -0.13944907 0.16665943 -0.09309288 0.15926978 -0.16728358 -0.08298269 0.07358788 -0.09445923 -0.24198854 0.030266196 0.009351595 0.10717588 -0.21386692 0.059285574 0.09191811 0.11173928 -0.009040234 -0.12588695 -0.1760949 -0.28650945 -0.019301226 -0.023242472 -0.2635927 -0.09322054 0.20938058 -0.014322888 0.0761847 -0.28455618 0.08160764 0.2539435 0.1948958 -0.19502778 -0.028940573 -0.11400827 -0.059045628 -0.2368307 0.13460729 -0.055199437 0.08975085 -0.14914508 -0.11258149 0.0025420468 0.11276945 0.11167213 0.039912377 0.037453003 0.22958378 -0.045956336 0.08068234 -0.16508006 -0.2511643 0.09124891 -0.10617208 -0.07927479 -0.22963996 -0.010815757 0.2400998; -0.12212032 0.21879692 -0.040005013 0.015716786 -0.2976992 0.19183518 0.26663473 0.05125175 0.37794903 0.06288354 0.0057424176 -0.013769247 -0.15800941 -0.15086475 0.16538423 0.13158582 0.2856623 -0.016460383 -0.010035288 -0.029398264 0.006498411 0.045508787 0.053609207 -0.026458204 -0.2604007 0.3894196 -0.012555636 0.010134872 -0.1550759 -0.39486134 0.11317617 -0.019149918 0.066656716 0.25519168 0.12634106 0.05439503 -0.18199486 -0.022308672 -0.18172716 0.12901859 0.1652745 -0.092502974 0.09747089 0.117724724 0.08836695 0.048012186 0.0719357 -0.041487318 0.21425737 -0.21181138 0.35395613 0.080925666 -0.12265222 -0.19337028 -0.047695104 -0.06369803 -0.021999856 0.05618313 0.24025752 0.034252226 -0.13444564 -0.27559185 -0.014502754 0.22285989 -0.06894537 -0.099502 -0.2743814 -0.20823942 -0.019133788 -0.040130466 -0.35622108 -0.12956458 0.062344328 0.15119514 -0.047297686 -0.12056394 -0.09482743 0.3670948 0.022183023 -0.057176657; -0.007996916 0.2483755 -0.123458974 0.15394072 -0.21246725 0.016411936 0.06331932 -0.26968834 0.17553286 -0.07450304 -0.22547607 0.17198715 -0.14819874 0.11294942 -0.11153281 -0.07914343 0.19432244 0.071495526 0.17320687 0.23328589 -0.1236665 0.206763 0.116232425 0.16896261 0.04158207 0.09969255 -0.17752221 -0.16723053 0.19264597 -0.062623955 0.023864187 -0.06729668 -0.023714552 -0.015141382 -0.026488375 0.053558227 -0.08498228 0.21432754 0.24534653 0.10786227 0.13680305 0.0772634 0.065743275 0.11417046 0.09215117 -0.025041183 -0.21999437 -0.117453836 0.078414924 0.02217584 -0.09347573 -0.059066743 -0.040104542 -0.12077872 0.000283028 0.20852908 -0.18485926 -0.16637106 0.031709187 -0.11083061 0.14978595 -0.20364507 -0.09039014 0.16231707 0.102082685 -0.1446034 -0.14591506 0.0015832835 0.11115963 -0.061156914 -0.22159936 -0.026441691 0.22949708 0.17914999 0.041884266 -0.07964706 0.13403562 0.049644068 -0.09087788 -0.12938046; 0.17213036 -0.19022821 0.17798235 -0.2091668 0.18063374 -0.20047908 -0.113983996 0.03455307 0.09599098 -0.1620407 -0.105938606 0.11585826 0.065182425 -0.1137942 0.047131997 0.17790936 0.06371309 0.00089247595 -0.091400504 -0.12202731 0.10482295 -0.1795458 0.21873583 -0.09107645 0.13479932 -0.16643474 0.032438967 0.17160441 -0.079926 0.13600323 -0.072521605 -0.13345231 0.013283874 0.14082697 -0.12941702 -0.11898231 0.16753955 0.14589761 0.06469121 0.057511955 -0.0033360405 -0.00581231 -0.15324987 0.07332557 -0.13778219 -0.15396275 0.13879912 0.08801286 -0.072406985 0.03026405 -0.1649565 -0.19645472 0.09682233 -0.021950027 -0.02221247 -0.04423698 0.1380324 -0.04852461 -0.08285793 -0.0075981705 -0.06188232 0.07173932 0.02416482 -0.20871611 0.09521521 -0.12060314 -0.06702168 -0.0066829626 0.10045609 0.22599837 0.24432369 0.16661769 -0.07153021 -0.11787665 -0.025063459 0.030370858 0.0517662 -0.07300604 0.13810144 -0.086738296; -0.112093054 -0.02059194 0.004684656 -0.12745965 0.073910534 0.037595753 -0.1407422 0.22355494 -0.17808357 0.14334352 0.0011027721 -0.10889888 -0.043915268 -0.23269181 -0.033752516 -0.15239175 0.07488016 -0.06912495 0.10879427 -0.09501724 0.12964289 0.060749777 0.016964626 -0.20515135 0.03622519 -0.19333495 -0.22088096 -0.036220077 -0.06472347 0.09106058 -0.083287634 -0.07960834 0.13636139 -0.11990942 -0.1608678 -0.24496506 0.11520922 -0.08411662 -0.037963428 -0.32375243 0.07235397 -0.22076094 -0.011238293 -0.028939314 -0.032316666 -0.0020314956 -0.16867565 0.004903509 0.15044872 -0.08714 0.16192766 0.28896427 0.062338326 0.33017203 -0.13893916 -0.042672496 0.14274378 -0.05693353 -0.15503961 0.18638042 0.15566172 0.059033733 0.05784005 -0.14562248 -0.012898829 -0.18725473 0.1229296 0.035558462 0.06239071 -0.18805988 0.0035488599 -0.057090756 -0.07696629 0.111208566 0.039290328 -0.024842937 -0.27565518 -0.031610373 -0.31018877 -0.078307234; -0.13680342 0.36389148 0.05571172 -0.032207105 -0.22350995 0.17042412 0.22035861 -0.24316733 0.22358768 0.20948602 0.17736553 0.13503288 -0.26769534 0.3269203 0.033544157 -0.004840648 -0.03445789 -0.019380422 0.3070345 0.2302298 0.10726868 -0.028850866 0.16890539 0.019454867 -0.25882438 0.23110978 -0.124371715 -0.037526634 -0.17191856 -0.4054662 0.32200205 0.20519653 0.2568875 -0.012399635 0.104657926 0.24466173 0.038535252 0.13802889 0.16830702 -0.024110267 0.041722797 0.028371373 0.21895432 0.057724714 -0.04113604 0.10608713 0.24128826 -0.20119806 0.0899265 -0.06303848 0.08300951 0.13046652 -0.14295039 -0.08129089 0.061219215 -0.13388483 -0.1633694 0.12858704 0.023690734 0.037185375 -0.2133037 -0.047739916 0.079413414 0.31713614 -0.03567772 -0.15978695 0.10128354 0.06524754 0.09853339 -0.20871001 -0.26725417 -0.14854194 0.23851186 0.2495532 0.09860756 0.16071093 -0.040952325 0.2523512 -0.1404024 -0.26263705; 0.16761242 -0.04534693 0.1421517 0.077511966 -0.07019019 -0.18927754 0.049541984 0.19450939 -0.21787953 0.11707853 -0.041011356 -0.16731383 0.14706035 -0.08781592 0.02575384 -0.04168074 -0.0911252 0.07021499 0.036649294 -0.22350653 -0.10401406 -0.095829405 -0.1750635 -0.18375508 0.0073099826 0.08583203 -0.16141325 -0.11099713 0.07446419 0.268312 -0.23437525 -0.12495104 -0.040379755 -0.15741353 -0.09511396 -0.15961531 0.14481628 -0.16652934 0.112261 0.022851203 0.08133918 0.12078136 -0.28127852 0.1048275 -0.10587923 -0.2911681 -0.18877351 0.15287411 -0.12744187 -0.18498228 -0.21286473 -0.067089096 -0.031256646 0.027763845 -0.052231852 -0.10614874 -0.036453795 -0.05233551 -0.26816526 -0.19975094 0.16050856 -0.12455894 -0.034323633 -0.102328405 0.116112106 -0.12069635 -0.16597219 0.17768049 -0.1970419 -0.03574884 0.13454199 -0.10117565 -0.11029541 0.035984743 0.2104498 -0.14104797 -0.10778827 -0.080474414 0.02918309 0.15969914; 0.05645319 -0.18448377 -0.26474732 -0.028170886 0.07416334 0.087685525 -0.15875122 -0.043550503 -0.058234744 -0.16615874 -0.043737557 -0.19548333 0.121097066 0.12845348 -0.07748981 0.107398145 -0.08964136 0.13177049 0.07720186 0.06189367 -0.19829585 0.08292522 0.025744801 -0.054829966 -0.15433313 -0.37311798 0.2570131 -0.077679835 0.14801666 -0.05996186 -0.0002878725 -0.12340661 -0.24128087 -0.21060997 -0.083842546 -0.40759704 -0.08324609 -0.24650505 0.1327939 0.08848876 -0.058059756 -0.108457014 -0.07046144 0.19889966 -0.01500905 -0.09299359 -0.15215959 -0.14709653 -0.23607402 0.15797208 -0.119820885 -0.012503298 0.26550978 0.1195434 -0.27503693 0.09008134 -0.047246616 -0.118945844 -0.12727618 0.06823666 -0.26133364 0.10943431 0.06681003 -0.31700715 0.1606781 -0.027904214 0.2967818 0.056337103 -0.022430029 0.24488443 0.08490614 0.0051371953 -0.012839406 -0.2479993 0.21744187 -0.14965577 0.1554904 -0.17074211 0.15262744 0.2999621; -0.06800409 0.23896302 0.29435298 -0.02518466 -0.27466923 0.12230096 0.18471715 0.1246442 0.13335441 0.16520704 0.2064257 0.10442477 -0.30720505 -0.10915719 0.2052321 -0.21497722 0.11749212 -0.07387764 0.19239683 0.2593607 0.109866954 -0.2479282 0.13476586 0.22701605 -0.2723854 0.3625014 -0.054671593 0.27857482 0.004762296 -0.24327394 0.0057347175 -0.04814502 0.055279613 0.23864955 -0.19023912 0.14043629 -0.30409724 0.26134706 -0.062825195 -0.29332528 -0.09394485 -0.20073979 0.198977 -0.30255628 -0.021379452 0.14227325 0.10275211 -0.32519066 0.1649952 -0.2043972 0.26168752 0.049009856 -0.19661438 -0.03876748 0.26646122 -0.14953159 -0.2330568 -0.15908627 -0.13361847 -0.084312834 0.012319855 -0.116864726 0.015012806 0.09270644 -0.024942905 -0.23513427 -0.18564977 -0.15682602 0.30105445 -0.20137939 -0.35409972 -0.15921992 0.34576046 0.17643662 0.013694095 0.34207585 -0.16882451 0.26455107 -0.3461778 -0.13498989; 0.017090833 0.05071745 0.1662502 -0.16424955 -0.17516859 0.024751587 0.047534276 0.035869986 -0.021352813 0.2034397 -0.16174062 -0.051498417 -0.14128377 0.06163826 -0.088412136 0.028300859 0.19572262 0.11562467 0.03541908 0.0166526 -0.0779731 0.020184655 0.14716487 0.20884234 -0.04130281 0.21066453 0.092738114 0.237498 0.1886008 -0.18342634 0.31336385 0.22519653 0.21241027 -0.021167273 -0.00675249 0.264559 -0.02219116 0.2241218 0.10438203 0.12405623 -0.0037733405 0.15768851 -0.0325159 0.14870164 -0.056892842 0.24991453 -0.12013257 0.029886803 0.2813877 -0.0318585 0.2088724 0.16497228 -0.012132427 -0.22521865 -0.056892164 -0.030871863 0.054472275 0.14566906 0.14706509 -0.08729378 0.018295445 -0.24461874 0.04381667 0.02891112 -0.029029913 -0.049418166 0.17739263 -0.13239147 0.16761012 -0.26506653 0.039538443 0.037088737 0.03112151 0.27329445 0.08272844 -0.19405574 0.1698348 0.1029555 0.096730225 -0.2450526; 0.142804 -0.015925556 -0.18344434 -0.057453025 0.05087434 -0.05824338 -0.12496818 0.21603756 -0.026909504 -0.2391982 -0.12431561 -0.19398788 -0.055931542 -0.14512521 -0.07592643 -0.06378715 0.07368266 0.0049236007 -0.17409159 0.00262776 0.20827128 0.12272477 0.11155116 0.010150467 0.27237174 -0.094964825 0.16054875 -0.24352156 0.11345944 0.025740083 -0.11809133 -0.18197 0.003471219 0.011308063 -0.019673979 -0.07313267 -0.07838922 -0.0052451347 -0.1033082 0.05057199 0.16642481 -0.059992615 -0.0051006232 0.15007082 0.058363065 -0.33377272 0.14711155 0.14004548 -0.0454799 0.0015933671 -0.1757541 -0.017824259 0.10939424 0.15490367 -0.051124312 -0.018480033 0.1376952 -0.10513426 -0.20903678 0.21634282 0.13249427 0.07677952 0.11106218 -0.11960272 0.044796642 -0.05568308 0.05229771 0.19732052 -0.10762465 -0.024005683 0.20844147 -0.11097455 -0.16902782 -0.2834826 0.19604714 -0.045503598 -0.17425029 -0.14253089 0.020487133 0.051982477; -0.049252983 0.14237335 0.10345522 0.026223991 -0.059478935 -0.024274644 0.06765714 -0.22554831 -0.1473088 -0.026112875 -0.010844943 0.1889851 -0.046733804 -0.1675026 -0.0043382403 -0.03310927 -0.11514046 0.033906817 0.15570049 0.040713526 -0.15443121 0.1829874 -0.032027412 -0.008939596 0.049814023 0.15211555 0.08736677 0.11440341 -0.050352305 -0.11222201 -0.07299435 -0.09839191 -0.0869718 0.14598112 0.07338115 0.17359042 0.12281414 -0.046791956 -0.094380274 0.18120317 0.07914102 -0.047153644 -0.04444633 0.045637786 0.17448892 0.19039412 0.060064554 -0.26048553 0.09345476 -0.0435854 -0.006741179 0.0030510828 -0.059509177 0.20354946 0.091538444 -0.010138554 0.07090221 0.07596385 0.14003208 0.03908176 -0.026983721 0.1463623 0.072762854 -0.13104664 -0.09911998 -0.14712815 0.071869396 -0.07245246 -0.17264253 -0.14288834 0.12436471 0.2055819 0.09983701 -0.18665172 0.14494202 0.03574111 -0.024364842 0.097333975 -0.015627602 0.08886031; -0.05562381 0.10928082 0.12019393 -0.012511099 -0.17599256 0.031033337 0.03560807 -0.23995058 0.22308111 0.20210576 0.20782197 0.19276406 -0.041869022 0.010476759 0.25119713 0.103950985 0.1436867 -0.09964173 0.28009948 0.034773864 -0.11771448 0.037807263 0.18876795 -0.058324225 -0.14045207 0.22191401 -0.028224846 0.12172691 -0.019870965 -0.1816014 0.20287737 -0.016439341 0.02331582 -0.13377243 0.013082225 0.06464541 -0.047278125 0.22994485 -0.09671114 0.20671646 0.10697782 0.16512841 0.29792333 0.015041188 -0.024236964 0.09110917 0.20373483 -0.20979464 0.3639476 -0.18169329 0.33964768 0.076821856 -0.0063156174 -0.20062827 0.10828138 0.047703937 -0.16103125 0.09020465 0.24773481 -0.07884057 -0.17745353 -0.10667349 0.093547635 0.14403935 0.1641548 -0.23367776 -0.17970937 -0.20847453 0.10429142 -0.2677069 0.034736924 -0.04222764 -0.030851584 0.2780424 0.014738623 0.03968041 0.15925442 0.25365326 -0.14489776 -0.29039192; -0.08953455 -0.0034193606 0.16231707 0.10053918 -0.016858155 -0.25981984 -0.002995145 0.18251832 -0.03078518 0.062190257 0.089498855 -0.019481396 -0.06417347 -0.03302984 -0.1196309 -0.04084447 0.066968754 -0.18779153 0.07076931 0.07544356 0.03649306 -0.023933537 -0.08796738 -0.047279578 -0.04227783 -0.12929742 0.03655095 0.0046319277 0.01626801 -0.180916 0.15766554 0.03699431 0.22761165 0.20499878 0.04087319 0.17670776 -0.16098237 0.1720276 -0.0056614233 -0.11264165 -0.14810695 -0.07064574 -0.22711638 0.039419226 0.19312957 0.020216778 0.17391284 0.22849423 -0.015962543 0.05032443 -0.14539985 0.11596759 -0.1118727 -0.060062557 -0.018690847 -0.04824241 -0.14948449 -0.06826967 -0.013111079 0.10967085 -0.08264702 -0.23216751 -0.14290002 -0.15187985 -0.19263436 -0.04344497 0.046189446 0.09480553 -0.20544505 0.16055816 0.033620868 -0.1873099 -0.02235854 0.03822943 0.09363615 -0.10968535 0.051676564 -0.17605771 -0.101186335 0.096266694; -0.31231079 0.23818454 0.3903919 0.017214492 -0.413551 0.12797306 0.46005273 -0.23178847 0.48260364 0.0003041199 -0.10152592 0.039485794 -0.27515605 0.10786876 0.038636394 -0.17984407 0.294139 0.1017364 0.29820243 0.28281084 0.063476585 0.07209941 0.12742725 -0.024092255 -0.42009124 0.24624662 0.047477428 0.0110941185 -0.06919513 -0.2540548 0.31600475 0.07059851 -0.0033877522 0.17383692 0.16516176 0.11730406 0.05095401 0.20283699 -0.19755784 0.09693682 -0.1991733 0.0067290436 0.14705427 0.13788475 -0.10135226 0.32230973 0.16255382 -0.29505295 0.3347342 0.07301092 0.118971676 -0.002687748 -0.010120253 0.05358955 0.025146246 -0.17868288 -0.24696347 -0.27324522 0.02712785 -0.25195593 0.10749948 -0.3738596 -0.021593232 0.084256515 0.08690199 -0.29054436 -0.091924064 -0.15692408 0.2237077 -0.39401978 -0.08065519 -0.105460376 0.2619096 0.38335246 -0.014671665 0.23701559 -0.10119663 0.0746654 -0.15581864 -0.36669484; 0.21320453 -0.027502025 0.14414717 -0.06601144 -0.103111334 0.108483054 0.026837774 -0.016149417 -0.1401399 -0.09008097 -0.113048635 -0.03766175 0.22552939 -0.0348828 0.0026846686 -0.1604852 -0.11799928 -0.031070847 0.24904472 -0.1070472 0.12480537 -0.16736421 -0.010920212 -0.16574155 0.08252056 0.0017461209 0.16718064 -0.09584892 0.19305675 -0.07327752 0.119214274 -0.15457779 0.06606175 -0.10685406 0.1000033 -0.03289489 -0.12586963 -0.0998554 0.082992695 0.011776215 -0.03796916 0.070064925 -0.011043673 -0.10409532 -0.20646298 0.18053369 -0.08586237 0.020706609 0.07907577 0.07530658 0.16658323 0.18250974 -0.1094736 -0.08917146 -0.08569385 -0.13116877 0.17773068 0.09585507 0.013890811 -0.19205138 0.00023656694 0.050638393 0.08736868 -0.0029346105 -0.09165855 0.10049787 0.18842371 0.08664027 0.018166833 -0.17334229 -0.13276108 0.09429859 -0.15695152 -0.09602559 0.109689735 0.076140024 -0.087412804 0.043203715 -0.16989785 -0.029090656; 0.16601358 -0.17299077 0.1672374 -0.017188754 0.29945645 -0.19961473 -0.19192085 0.017280404 0.13727741 -0.07380099 -0.12924296 0.15138195 -0.044843696 0.02538433 -0.08122131 0.058222797 -0.16434683 0.10514239 -0.0967182 -0.2208941 -0.032848723 -0.12913494 0.07855599 0.1967536 0.058337525 -0.26934862 -0.04324894 0.14521009 0.0036519908 0.052737877 0.014459885 -0.20786913 -0.07768137 -0.021561993 0.08233899 0.07427015 0.11417865 -0.009004603 -0.09220823 -0.021288443 0.11545885 0.1261622 -0.11922432 0.06426883 -0.078368545 0.020710893 0.010187293 -0.052341934 -0.14466849 0.11891761 -0.058717176 -0.123217694 0.14808433 0.07381681 0.16962159 -0.099269144 -0.0065090815 0.16149822 0.052289546 0.15815042 -0.0044462383 0.08238282 -0.20502314 0.008171572 0.043487728 0.0666074 0.023093708 0.13338381 -0.09467155 0.014292556 0.3166272 -0.17683525 -0.078201875 0.160647 -0.1784527 0.046518706 -0.13235274 -0.24684922 0.08821995 0.15220256; -0.12550962 0.039513305 0.057449188 0.026022451 0.079533055 -0.21149698 0.04748648 -0.081886195 -0.27063027 0.044870507 -0.050409388 -0.05993746 0.02860655 -0.013721314 0.08526114 0.08788662 0.17837778 0.1350306 -0.028372806 -0.13510393 0.13608418 0.011200648 0.11218824 -0.17400663 0.17785017 0.033713847 -0.020181762 0.03843097 0.09152641 0.079140805 -0.18850552 0.042921904 0.18725486 0.10289625 0.091872774 -0.047047265 -0.16579689 -0.010873852 0.06618864 0.051316775 -0.044465076 -0.11036443 -0.14148045 0.028612068 0.042007104 -0.22580607 0.0722394 0.11070258 -0.29773474 -0.035015468 -0.15592405 -0.25562373 0.114291094 -0.10997065 0.08130171 -0.092485845 0.02556451 -0.12467068 -0.09576431 0.24213539 -0.0650458 0.10807329 -0.19101934 -0.2494376 -0.08740641 0.19506493 -0.06625031 -0.005642432 -0.0817639 0.17678939 0.004506108 0.1504697 0.025313884 -0.038687836 -0.17627244 0.10823173 -0.0014425619 -0.07643367 0.27089038 0.024025034; 0.023342371 0.1876275 0.14159654 -0.10256818 -0.21721905 0.28704235 0.4003631 -0.18368325 0.20862813 0.1322155 -0.061137177 0.2650918 0.068873435 -0.15724126 0.17856944 -0.21246682 -0.06050767 -0.15721318 -0.06964749 -0.01409026 -0.026842456 -0.13405201 0.10268169 0.2552553 -0.19569091 0.36707106 -0.040317442 0.049465187 -0.2069384 -0.13579917 -0.070193104 0.16668378 0.10104117 0.17331812 0.082418114 0.14135893 -0.1796088 0.2555804 -0.10489268 -0.15303992 0.2063056 0.115032434 0.017917333 0.024607712 0.23131381 0.18281743 0.143604 0.11327155 0.023536852 0.041974317 0.22309731 0.24024211 0.046346534 -0.10960233 -0.024121765 0.039779827 0.0146739185 -0.13626797 -0.018530255 -0.038006134 0.21453206 -0.2084494 0.109721884 0.3916992 0.10192467 -0.11112275 -0.27817613 -0.26691252 0.024938712 -0.13970128 -0.19173768 -0.053353112 0.12678206 0.24510458 -0.070495464 0.1353782 0.012053966 0.24314998 -0.09646903 -0.24237533; 0.13571724 -0.09527922 -0.18807276 -0.17550911 0.1721044 -0.012304307 -0.10703638 0.11811067 -0.21674101 -0.17097071 0.103787385 -0.11319553 -0.10606686 -0.19718917 -0.025734162 0.12064127 0.048147388 -0.03927355 -0.38330457 -0.042259417 -0.013714796 -0.14592932 -0.16606067 -0.2041849 0.20746154 -0.1245754 -0.013088208 -0.17579758 0.15254344 0.4477575 -0.09733704 -0.28419945 -0.024063585 -0.07576245 -0.09852385 -0.38552752 -0.15384959 -0.07641646 -0.058484178 -0.20995477 0.20858924 -0.063083895 -0.26832604 -0.14160612 -0.08849442 0.0020037475 0.08840345 0.1959002 -0.36168322 -0.21615319 -0.11654801 -0.16719583 0.0434014 0.024104698 -0.012857763 -0.09794532 0.12920175 0.113711074 -0.2537353 0.27777845 0.09499172 0.11937903 -0.21392386 -0.2746841 -0.0022735836 0.088655256 0.046926484 0.09220796 0.0020180317 0.3351689 0.31945232 0.077966884 -0.38953152 -0.19513609 -0.16167472 -0.14035851 0.045219745 -0.10121323 0.2190567 0.2934375; 0.16860065 -0.28783616 -0.24244395 0.20718676 0.39851707 -0.1843666 -0.17595087 0.0683664 -0.28945753 -0.21253717 0.08695834 0.060974974 0.118182614 0.16344227 0.043655213 0.085509434 -0.09543489 0.05765708 -0.12217091 -0.088121116 0.13626787 0.16842274 -0.23464146 -0.2618491 -0.027103001 -0.16549233 0.18343756 -0.23500991 0.00503679 0.13983846 0.019798879 -0.008560993 -0.04526804 -0.116647415 -0.035575572 -0.36448124 0.10582469 -0.008961588 0.20032108 0.038929205 -0.18787408 0.18226604 -0.1035103 0.0063529983 -0.25282487 -0.008512843 -0.27779022 0.1261752 0.017514706 0.21306011 -0.23770283 -0.14176749 -0.062042836 -0.102965266 -0.23571853 -0.052794304 -0.077765875 0.025089765 0.13785116 0.12612185 -0.19813232 0.07868304 -0.11133467 -0.3274462 0.005938718 -0.03541807 0.1453438 0.17898293 -0.01360284 0.08261515 -0.006607595 0.04034944 -0.009729112 -0.25285852 0.140875 -0.12574326 0.15243722 -0.010531964 0.24143365 0.24718192; 0.25708917 -0.21623203 -0.03330184 -0.036602646 0.13812685 -0.039985072 -0.23452084 0.010514766 -0.07405728 0.045495305 0.027216233 0.051759146 -0.01150455 -0.060830478 -0.15134102 0.022915445 -0.1574471 -0.06161249 -0.1804083 0.035425745 0.13849428 0.14028865 -0.09037164 0.09409039 0.069084026 -0.25477454 -0.10492473 0.077322304 -0.11471029 0.07398938 0.00023922452 -0.24672467 -0.20693119 -0.08356514 -0.031006327 -0.27464417 0.1103586 -0.03126974 -0.021250095 0.1324281 -0.073590375 0.030248726 0.04715702 0.019831553 -0.10934523 -0.23509857 -0.11844099 0.17245848 -0.1446245 0.045904618 -0.2850565 -0.22597054 -0.015338899 0.02906787 -0.12034603 0.06403529 -0.021867568 0.017877586 -0.24380022 -0.049534615 -0.06571204 0.13857295 0.025708936 -0.14096127 0.158833 -0.09129502 -0.16265462 0.06883513 0.04570115 -0.019460222 0.1096678 -0.122228906 -0.19454803 -0.120389275 0.10048748 0.15124646 0.042145666 -0.15794472 0.19184849 0.057209704; -0.14205277 -0.2118179 0.16690534 -0.14099081 0.068460345 -0.05269024 -0.0050686346 -0.012851711 -0.16777194 -0.18535143 0.0045017577 -0.12927258 0.04169301 -0.007064311 -0.20510969 0.23203059 0.1018787 -0.1476334 -0.27127162 0.021095445 -0.01042733 -0.07405778 -0.08826165 -0.16811734 0.102567315 -0.086436175 -0.11777957 -0.11224589 -0.109793976 0.0016715359 -0.18552217 -0.00063507434 0.10252954 0.02132162 0.027354067 -0.09864075 0.10692472 0.11334418 0.047465578 -0.14422122 0.025060903 0.18894374 -0.22772364 0.041906018 -0.079536356 0.04399933 0.15062553 0.07142315 -0.05377943 0.12835592 -0.13093998 -0.08796526 -0.07559486 0.20682406 -0.15980867 0.0652792 -0.21095112 -0.08300189 -0.0005014506 -0.043831512 -0.16088137 -0.0056144185 -0.15592884 -0.060200617 -0.13810338 0.12067103 -0.017911134 -0.024541084 -0.11413452 0.22447096 -0.014098192 -0.13798267 -0.09711649 0.04631431 0.08205752 -0.16239643 0.042686798 -0.19165044 -0.07729581 0.19286063; -0.028088786 0.034355093 -0.13750209 -0.16067904 -0.3050576 0.05018338 0.1502918 -0.167737 0.05317258 -0.125241 -0.09816441 0.038007163 -0.12507272 0.0608976 0.033063788 0.026010394 -0.052221384 -0.16552392 0.05156991 0.16884471 0.074810684 -0.041794755 -0.16569665 -0.18378633 -0.33243567 0.17635074 0.051378716 -0.11925435 -0.15740448 0.00546582 0.19133931 0.048599277 0.08862822 -0.1298628 0.13602473 -0.14990771 0.13613468 -0.06362535 -0.06829642 0.08963772 -0.008494898 -0.13466315 0.15736169 -0.12991385 0.078722976 0.10082257 -0.040887713 -0.03959069 0.29922745 0.045640558 -0.1055822 -0.1078332 -0.22688344 0.001495926 -0.082433484 -0.12275297 -0.19127588 0.17101143 0.067404754 -0.11802682 -0.08473981 -0.025818134 -0.022187391 0.13718578 0.051929954 0.040860396 -0.11042823 0.06775764 0.2007832 -0.21988125 -0.25586966 -0.0066312705 0.17047295 -0.02402559 0.13834207 0.18679132 0.06306086 0.234882 -0.18880123 -0.106731996; -0.30606678 0.15401419 -0.08269662 -0.063783616 -0.16991968 0.10621388 0.36607388 -0.12763894 0.0981735 0.17341869 -0.17681496 -0.01169805 -0.09864709 0.074528806 -0.036575865 0.06479172 -0.050032068 0.102799125 0.10928395 -0.0045341854 0.059657797 0.17878981 0.18480667 -0.002819436 -0.17376633 0.19643803 -0.1797356 0.13879119 -0.01655524 -0.31022868 0.088301264 0.20254833 -0.027671903 -0.09556016 0.16244939 0.05203721 0.19305319 -0.02686203 0.0633298 -0.04973461 -0.16253491 0.14847514 0.04352539 -0.028540451 0.19797312 -0.040335882 -0.026344797 0.020887144 0.25337052 -0.10704606 0.24116264 0.04535745 -0.14287172 0.044705953 0.10290636 -0.23258941 -0.23690207 0.064331375 0.06678651 0.044767037 -0.06726477 0.0100289285 0.09822937 0.07912549 0.036026604 -0.16104029 -0.18732934 -0.08416018 0.24958365 -0.05263663 -0.32001495 0.16084665 0.22727302 -0.030723587 -0.20496958 -0.18942429 0.13816334 0.3322224 0.04725803 -0.06882995; -0.1726425 0.22341374 0.04734939 -0.12212945 -0.2159572 0.06143577 0.33302453 -0.2134871 0.26023695 0.23548447 0.09866008 -0.022412688 0.034724884 0.05524412 -0.02992719 -0.10834501 0.013461425 -0.114734024 0.33722502 -0.010822864 -0.057981644 -0.18867728 -0.0929755 0.173814 -0.24249716 0.077641554 -0.11235389 0.27694622 -0.21871161 -0.22794795 0.29573652 0.12709615 0.08989827 -0.001760774 -0.08205238 0.10660046 0.043728158 0.23072456 0.18391587 0.08545987 -0.10579217 -0.21327391 0.06981971 0.04588684 0.14818767 0.21902792 0.28981405 0.024551636 0.27607006 -0.059328318 0.035331868 -0.066932574 -0.06029438 -0.23560384 0.32622617 0.021417834 0.0065169754 -0.123088926 -0.00058440655 0.13000308 0.14903817 -0.36357415 0.0018569223 0.019056723 0.06347036 0.062926 0.04038782 -0.22444087 0.25506276 -0.12867883 -0.15804586 0.0910966 0.1858957 0.2945045 -0.18161361 0.12009315 -0.08532949 0.28036848 -0.24589704 -0.13800354; -0.09578655 0.08150665 0.002755269 -0.14896396 -0.21578485 -0.046300214 0.22039129 0.09013739 0.14449252 0.21637991 0.15713485 -0.026288038 -0.1915305 -0.10080412 0.2193159 -0.19183616 0.09047799 -0.064423285 0.23135696 -0.11054083 0.065711 -0.07008346 0.22072598 0.2594585 -0.018883953 0.2865789 -0.07772406 0.29442194 -0.13501447 -0.21909952 0.16090557 0.16491438 0.08747326 -0.12636833 -0.033740144 0.19545306 0.11021498 0.054375563 -0.13162799 -0.16406928 0.2024471 -0.15292497 0.26719 -0.21800514 -0.022150839 -0.03477763 0.13927193 -0.20254447 0.19253151 -0.22959732 0.28507942 -0.040295422 -0.29536146 -0.056613885 0.19027868 -0.065637626 0.027334746 0.12886427 0.06483309 0.11199546 0.030453635 -0.11208458 -0.028137553 0.27672487 0.025795149 0.066691935 -0.24620536 -0.25492364 0.14531015 -0.17171478 -0.007518499 0.15893379 0.2016129 0.16719326 -0.25784907 0.026319891 0.08444025 0.21283543 0.037418026 -0.075093046; -0.21206257 0.17182007 0.04655821 0.0059114066 -0.23157619 0.16758971 0.10279314 0.05017437 0.0071335603 0.03393599 0.13789664 0.23016912 0.14907238 0.017661067 0.019823017 0.011834884 0.06785121 -0.08096558 -0.06256315 0.12920947 0.14304386 0.15319262 0.11042687 -0.17288516 -0.10536921 0.06312831 -0.019339345 -0.15539883 0.01414486 -0.03759666 -0.069269635 -0.029363947 0.21083745 0.12391063 0.03261001 0.29887646 0.19293374 0.077241085 0.18241398 0.20316455 0.03877482 -0.027006749 0.2977784 0.16150625 -0.0063603665 -0.040867582 -0.055139884 -0.16875625 0.19404267 0.18079612 0.11050822 -0.018449051 0.063075244 0.035133544 -0.0010595323 -0.15793797 -0.17793645 -0.18734595 0.1411354 0.10793246 -0.07584259 -0.076579936 0.06309085 -0.031778995 0.016498316 0.084062785 -0.15387872 0.16562298 0.21825443 -0.108721904 0.05957054 0.08931919 0.009314688 0.17558871 -0.033034448 0.119483866 0.18693234 0.022241727 -0.17453393 0.10644969; 0.15895893 0.05417436 -0.17461517 0.028859932 0.013096575 0.2925876 0.1465913 -0.008308337 0.09988469 0.17255029 0.08654641 0.056406535 0.14831346 0.21424863 0.0150336735 -0.27753475 -0.044030875 0.10848356 0.23570386 0.14088878 -0.14279318 0.094214424 -0.091421224 0.11169039 0.014244504 0.2935387 -0.16776899 0.135672 0.09296575 -0.17331083 0.08635585 -0.11690181 0.08142103 -0.10197304 -0.11816456 0.14601184 0.14674163 0.080257 0.05848768 -0.03494163 -0.064390026 0.014178203 0.17655885 0.0615694 -0.09897916 0.23987876 0.00937926 -0.0095097395 0.16399612 0.004199067 0.1584363 -0.017036954 -0.29963723 -0.18340738 0.010289654 -0.05037519 0.09865101 -0.13547611 -0.04744306 -0.004369719 0.10000183 -0.09761025 -0.06934583 0.022909638 -0.036437184 -0.08794391 0.08432556 0.12955283 0.19837144 0.015223591 -0.13550542 0.08592842 0.0027751427 0.124670655 -0.10373017 0.05988714 0.21848485 0.06358041 0.04628267 -0.062141392; -0.080838814 0.00060895795 -0.03950997 -0.13705924 -1.1021533f-5 -0.27348596 -0.19231626 -0.0076031005 0.04997738 0.11401261 -0.044056702 -0.1644296 0.059104007 -0.2120714 -0.16228049 0.012178386 -0.11700662 -0.22649093 0.0056993985 -0.048494168 0.16178341 0.12942246 -0.07298568 0.12598877 0.05149922 -0.14174108 0.069860525 0.1332616 0.16629784 0.16369791 0.060462467 -0.021982789 -0.08669195 -0.12682307 -0.086152494 -0.33507994 -0.14826235 0.098577686 0.13474508 -0.114149325 -0.022756815 -0.035390373 -0.03890595 -0.19984536 -0.043306172 -0.13959533 0.077972636 0.13436447 -0.03291507 -0.106773585 -0.17256275 -0.054061774 0.32400396 0.2041759 0.023320038 0.07081919 0.1409241 0.047116008 -0.085758545 0.13493259 0.028643103 0.23923942 -0.16525365 -0.29194692 -0.14738986 0.02954308 0.11508633 -0.09391319 -0.08233349 0.19670512 0.26288152 0.043467537 -0.2171651 -0.023485262 -0.13795646 -0.06684908 0.050384182 0.039369363 0.055669483 -0.059331313; -0.2626393 0.028539555 0.13533965 -0.1162689 -0.11304309 0.11651109 0.11435291 -0.20100726 0.01612832 0.054616887 0.014636265 0.29079503 0.0047034267 0.035198294 0.054687526 -0.013839371 0.14548154 0.014063984 0.093304455 -0.05098855 -0.080978 0.06939339 0.034366656 0.008858066 -0.01606912 0.23770343 -0.027202263 0.055355992 -0.09941216 -0.16386452 0.14320132 0.07946893 0.21882537 0.06727943 -0.12671551 0.07257624 -0.047655623 -0.10016512 0.21050158 0.20706913 -0.07013549 -0.08577721 0.24328849 0.19755098 0.117671594 0.043278925 0.12412976 0.044514302 0.16631849 -0.07849929 0.07647604 0.107671484 -0.097993605 -0.08298764 0.05411907 0.11033632 -0.065841354 0.08612422 0.19014895 0.013808924 -0.028087217 0.043966584 0.034692634 0.31683758 -0.16463782 -0.02712251 0.08444841 -0.13793477 0.16866377 -0.102055326 0.03429541 0.010801268 0.031961914 0.29129213 -0.08662183 -0.08243658 0.05701196 0.03162837 -0.079790585 0.0017920437; -0.1459078 0.0935507 0.0177106 -0.093497 -0.1529427 0.28385374 0.29799587 0.10149733 -0.05619935 0.1205322 0.15841165 0.15342684 -0.18134753 0.10327518 0.19362311 -0.0800326 0.046984743 0.069662035 0.0076390263 0.20796789 0.12059932 0.13840282 0.116005 -0.15129623 -0.21062846 -0.0714279 -0.01474569 -0.05072694 -0.06061681 -0.022149263 -0.058932867 0.19659151 0.22222738 -0.1875632 -0.06562384 0.058844574 0.024172088 -0.019129613 -0.050797336 -0.02890693 -0.048258826 -0.16765113 0.012750057 -0.027063098 0.06844234 -0.031190082 0.0012701737 -0.16330399 0.26239383 0.018598335 0.233958 -0.08747643 -0.04436505 -0.1636944 0.19134288 0.092071295 -0.063135676 -0.17790648 0.10403131 -0.14588282 0.093778625 0.059517957 -0.085987344 0.1325777 -0.054628477 -0.20611651 -0.19547458 0.011985524 0.012170335 -0.18949667 -0.24850795 0.11334665 -0.036773324 0.063563526 0.1326167 0.21263474 0.011172056 -0.058827024 -0.036444996 0.07204245; -0.31194764 0.25129855 0.09334893 0.03235868 -0.27057216 0.15568723 0.088909395 -0.17518622 -0.00047557117 0.15367562 -0.044297677 0.065579586 -0.106933996 0.13845508 0.16300322 0.16197138 -0.056735948 0.08357611 0.23937416 -0.056884937 -0.07439751 -0.14728272 -0.05625785 0.3238912 -0.037358947 0.053455792 -0.21842428 0.13039184 0.01307832 -0.32607442 0.15551308 -0.149429 0.3787696 0.116028115 -0.17234296 0.18496913 0.08840354 0.16043949 -0.16626586 -0.20591065 0.10812725 -0.08212912 0.29661915 0.07732884 -0.025689244 0.17182556 0.14776847 -0.12035313 0.06016274 -0.21363622 0.2582919 -0.08476417 -0.23250887 0.020397168 -0.03061392 -0.061446536 -0.055427983 -0.013966375 -0.07129468 -0.12592281 0.16422097 -0.09891353 -0.009065142 0.30520657 -0.029667018 -0.020428196 -0.17498654 0.05458506 -0.002948878 0.064294174 -0.042957895 -0.10978312 0.035586935 0.0869168 0.023681698 0.053496182 0.0065547377 0.062100556 -0.04886666 -0.1785568; -0.009205318 -0.017887102 -0.21776381 -0.0064501897 0.19388959 -0.11287662 -0.15847638 0.07466201 -0.19083086 0.0059539317 -0.21245944 -0.07173308 0.101678155 -0.1585451 -0.1536894 0.070452124 -0.12117553 0.057995237 -0.12503117 -0.14363423 -0.073573574 0.07533716 -0.008304565 -0.049219638 -0.1988507 -0.27556685 0.12459143 0.027987609 0.1557867 0.01701807 -0.21877564 0.01933494 -0.29114383 0.12481763 -0.06416751 -0.1662757 0.12990768 0.08253869 -0.023762373 0.036778912 -0.07710095 0.20888215 -0.080436096 -0.09575751 -0.020684985 -0.120530315 -0.16800398 -0.16299278 -0.2009283 -0.048191532 0.05527006 -0.19865634 0.09577155 0.127951 0.033094976 0.104620405 0.09507521 0.021057142 -0.086881965 -0.07968745 -0.10413869 0.12192098 -0.06131662 0.073774494 0.040037584 0.17714268 0.15366136 0.1431593 -0.20073803 0.12050496 0.1870559 -0.083878964 -0.16758497 -0.19066788 0.10580194 0.054063123 0.18154855 -0.24124646 -0.030027533 -0.06599814; -0.23958112 0.14193244 0.13107337 -0.010606055 -0.4167503 0.033795062 0.33081925 -0.0054320507 0.3049282 0.080505036 -0.16716377 -0.006920686 -0.22517638 -0.15162219 0.25997087 -0.35089275 0.022903994 0.025438303 0.2699135 0.09323813 -0.09658973 -0.08981603 -0.11382716 -0.05009245 -0.08763871 0.21818751 0.0276461 0.018448776 -0.020965427 -0.11090745 0.2314312 -0.026175853 0.09882012 -0.013375907 0.07424146 0.25571314 -0.1113302 0.031716406 -0.1562165 0.024710773 0.090885766 -0.10791936 0.14950086 -0.08373418 -0.11998948 0.3577611 -0.025640566 -0.0015668158 0.22410502 -0.09479024 0.34612256 0.018445736 -0.41019744 -0.24803218 0.13530272 0.08658652 0.020766417 -0.06492343 0.021315759 -0.054318465 0.10118628 -0.17842072 0.1940811 0.3298469 -0.09141182 0.016526522 -0.13625209 0.049261916 0.20202324 -0.30025935 -0.083188154 0.1458862 0.24280019 0.010848999 0.02790712 0.06921091 -0.02023121 0.23534149 0.036856517 -0.25559986; -0.038541894 -0.10380106 0.16266842 0.024523359 0.25485715 0.0018463716 0.054420494 0.11812074 -0.19792776 -0.11284791 -0.048265073 -0.17367308 -0.051886134 -0.19186538 0.09144675 0.16375051 -0.050267838 0.053787787 -0.11655161 0.0059653693 0.11979297 0.045161776 0.12565438 0.051466614 0.24707311 -0.11274111 -0.006869656 0.036851145 0.17358945 0.22734372 0.07150605 -0.081389576 -0.003358394 0.036536373 -0.17090331 -0.22996859 0.18071334 0.012518015 -0.06030771 -0.09906528 -0.06098949 -0.009651393 -0.1070654 0.07270504 0.1589923 -0.18219635 0.11875407 0.009039306 -0.01570772 -0.13293396 0.041983265 -0.114216216 -0.012456721 -0.04091038 -0.007457246 0.13933662 0.061838403 0.023598148 0.1245822 0.17306638 0.13215357 0.13660657 0.04443286 -0.24092624 -0.015263341 0.16804941 -0.090551995 0.068447635 -0.120965496 0.1954822 0.112979 0.038252406 -0.07107842 -0.19023208 -0.12390671 -0.21413298 0.09172388 -0.17477702 0.20078373 0.1079434; 0.08418363 -0.2443407 0.016467309 -0.0700646 0.093761794 -0.0771559 -0.21026526 -0.017587764 -0.085187994 -0.108786695 -0.020384127 -0.0956994 0.040841203 -0.1272013 -0.17673574 0.20741948 0.03440024 -0.23459227 -0.20580643 -0.1653329 0.005738285 -0.024209758 -0.104733005 -0.06235242 -0.006724246 -0.1526161 -0.075849295 0.0020469513 -0.010268096 0.21463753 -0.2915382 0.0060858484 -0.05032533 -0.029517082 -0.1625904 -0.22945537 -0.10717426 -0.1635927 0.084011465 -0.04182705 0.010294153 -0.086027294 -0.27805924 -0.03123632 0.10901038 -0.013123427 0.022720013 -0.0038522352 -0.27301568 -0.10646346 0.003136223 -0.09175483 0.24053563 0.0327659 -0.0022249776 0.0011801433 0.024820322 -0.12867236 -0.20479435 -0.1054759 0.001700946 0.1357296 -0.15025012 -0.30795977 0.14934468 -0.026291426 -0.09567436 -0.07264837 -0.04067069 0.17395583 0.14645657 -0.046360943 -0.29183602 -0.22367758 -0.12031464 0.117729224 -0.124656186 -0.052602015 -0.053896386 0.24357867; 0.1781293 -0.028412359 -0.11593253 0.13459545 0.113946654 -0.22116026 -0.22777444 0.2038492 0.02319851 0.17584817 0.22168654 -0.08837251 0.13354333 -0.08388918 -0.09849681 0.07971565 0.012442243 -0.16484272 -0.112378195 -0.03151313 -0.035926856 -0.058831025 0.14050896 -0.051886074 -0.010351981 -0.24834964 0.06630053 0.054576404 0.13531356 0.028321836 -0.2537503 0.04512483 0.21951129 0.048906278 0.08152713 -0.17093244 0.00047074602 -0.15539137 0.109860405 0.14347962 -0.096382655 0.051679537 -0.22477183 0.0300319 0.18139611 -0.18225756 0.13756302 0.13636844 -0.059585556 0.006497147 -0.009914647 -0.15278327 0.16160166 0.048575744 -0.086567245 0.124708086 -0.13211744 0.06666246 0.06707177 0.20008388 0.0468611 -0.040886477 -0.10306894 -0.24371988 0.023007298 0.085707314 -0.092241585 -0.15640172 -0.04001117 0.111550055 -0.043079816 -0.027330395 0.033410553 -0.16210566 0.07008229 -0.19484861 -0.1254544 -0.28975296 0.115711495 0.13716705; -0.12146534 -0.17391308 -0.13020349 -0.048675243 -0.1489741 -0.1346317 -0.02802187 0.048319753 -0.09383058 -0.0008469641 0.120285444 0.04167408 0.172672 -0.14073355 -0.124110654 -0.17543638 -0.10912466 -0.10224221 -0.12279707 -0.038633734 -0.016915495 0.11391324 0.023032146 0.11678941 -0.110649824 -0.18294331 -0.08044364 -0.11217313 -0.082128815 0.15120548 -0.10797091 0.1265919 -0.21311365 0.20670821 -0.023020813 0.107948184 -0.23612064 -0.06219808 -0.05349681 -0.008459892 0.1428725 -0.05186359 0.010859054 0.1249676 0.13365352 0.028224258 -0.0031645938 -0.06504221 0.1863562 0.10558952 -0.17693786 0.00967176 0.18691283 0.075396344 0.029395444 -0.016452497 0.14152701 0.07688613 0.046250515 -0.0036929958 -0.01757461 0.015880218 -0.02803087 -0.24333382 0.03909389 0.0013746717 -0.029035076 0.14703543 0.19820237 -0.18846619 0.17080703 -0.15765893 -0.2146391 -0.054667395 -0.00268636 0.012129329 0.010652452 -0.14427648 0.1469106 -0.05714779], bias = Float32[0.13431093; 0.023000194; -0.07675651; 0.091166; 0.2454918; -0.1100163; 0.015007818; 0.031706564; 0.04711999; -0.18355925; -0.47458175; 0.2442546; 0.098089255; 0.30417567; -0.04854727; -0.022177367; 0.27510926; -0.19191861; -0.15489538; 0.08831381; -0.19238466; -0.1067386; -0.17605633; -0.046287432; 0.2506656; 0.05676555; 0.30697152; -0.21764666; -0.026855588; 0.29874304; -0.40434825; -0.020671416; -0.13691919; -0.5264583; 0.0004979563; 0.2845575; 0.023595449; 0.0037392704; 0.002496946; 0.106395364; -0.43063727; 0.009965995; -0.08135435; 0.19262713; 0.012138069; 0.17219572; -0.16673099; 0.11964542; -0.08172709; 0.0477371; 0.083006874; -0.012478583; 0.2848326; -0.014353997; -0.047946215; -0.2543947; 0.034411114; 0.066647485; 0.025234297; 0.082879886; 0.26420552; -0.08612804; -0.032236923; -0.22893816; -0.0511812; 0.03433021; 0.1402352; 0.0415261; 0.093253694; -0.0524934; -0.10992003; 0.019715022; 0.14007238; -0.019675309; 0.061419986; -0.25507134; -0.2630254; 0.026873644; -0.062421024; -0.33235818]), layer_3 = (weight = Float32[-0.09808804 0.0152480975 -0.13278265 0.06901161 0.065014414 -0.033436466 0.10759012 -0.23372827 -0.0087169325 0.025486892 -0.1635552 0.0940887 0.18249893 -0.033535726 -0.031174444 0.04840875 0.18785019 0.07168416 0.018139206 -0.020794824 -0.01998877 0.06113359 -0.055664625 -0.031115893 0.14954549 -0.16448672 0.009046954 -0.05326839 -0.044733196 0.07162114 -0.08820893 -0.11557681 0.0010462487 0.10112242 0.0731994 -0.048427474 0.067863256 0.034532357 0.16763149 -0.16118303 0.080761015 -0.01791464 0.12090871 -0.040320665 0.07718932 0.050234236 -0.105129436 -0.22054341 -0.18014453 -0.14847705 0.09081674 -0.03626102 0.1435782 0.092702374 0.12998489 -0.11124086 -0.07172513 0.032010943 0.12105052 0.05323745 0.13566273 -0.1371921 -0.1213865 0.0027235588 0.023388898 0.06928609 -0.010604629 0.11218448 0.09435119 -0.00081184175 0.03573871 0.1543584 0.0005350688 0.034029923 -0.08561136 -0.1708902 0.12244903 -0.02797942 -0.00742932 -0.09436928; 0.062193733 0.41551185 0.25018156 0.22513914 -0.6732328 -0.076340295 -0.46313807 0.54123074 0.089951575 -0.53509027 0.45988438 -0.022750111 -0.36665377 0.38630953 -0.0015875838 -0.44419166 -0.090813674 -0.33782396 -0.4750817 -0.05021227 -0.26471126 0.61493427 0.21783683 0.13431239 -0.04362583 0.16682747 0.02374861 -0.03095487 0.045553092 0.21146345 -0.024722096 -0.44340485 -0.5694966 -0.42468125 -0.5173929 0.47476023 -0.47326666 -0.20958108 -0.6515509 -0.5015902 -0.56410563 0.5380638 -0.21831216 0.21272318 -0.62984675 -0.018304605 -0.14392246 -0.3340636 0.31155902 -0.6582638 0.06482795 -0.29590923 -0.11790575 -0.5781102 0.061844643 -0.7240355 -0.009518042 -0.49885315 -0.59104526 -0.18816546 0.08886662 0.37387478 0.026612112 -0.2849315 0.48477548 -0.14779158 -0.069746405 -0.3797999 0.45681658 0.4944214 -0.38163036 0.1771661 0.22972551 0.11325924 -0.2455555 0.15799297 -0.21924439 -0.27991098 -0.5718325 0.33972958], bias = Float32[-0.1402916; -1.5102644]))
loss_neuralode(p_trained_nonoise)

#Moredate noise

rng = Random.default_rng()
Random.seed!(99)

#Adding moderate noise to data:

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 7e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))


#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)





prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-80]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained_moderate = (layer_1 = (weight = Float32[-0.54207605 -0.08931858; -0.59132355 -0.044312436; 1.0597245 -0.2904367; -0.8415495 0.2518355; -0.9635182 0.34537458; 0.7465145 -0.16712336; 0.08815579 0.0463996; -0.42038482 0.43479463; -0.47389007 0.038572628; -1.0732368 0.3044327; 1.0243837 -0.24159656; 1.3783909 -0.36483982; -0.90717334 0.049350295; 0.24813363 -0.10815106; 0.7474638 -0.09410347; -0.39710385 -0.036752835; -0.87189215 0.2045615; 0.8214552 -0.39658958; -1.299256 0.087683894; 0.26793545 -0.12529227; 1.1832998 -0.33983886; -0.94385207 0.15146747; -0.329743 0.06348516; -0.43338436 -0.017511755; -1.2315717 0.30009922; -1.2831876 0.339342; -0.5267163 -0.079912215; 0.5056784 -0.31040782; -1.4039207 0.3399155; -0.75088274 -0.013748293; -0.20255332 -0.27451023; -0.019680737 -0.11039811; -1.3249797 0.48787087; 1.2282455 -0.42017147; -0.5807599 0.26207042; -0.2811132 -0.09922412; -0.9019461 0.17038432; -0.41125125 0.0017450065; -0.4238288 0.26609135; 1.0131577 -0.3712033; 0.98539764 -0.55725086; 0.8064437 -0.20428264; -1.5322368 0.43054503; 0.32411274 -0.07060059; -0.41011345 0.4305979; 0.56363606 -0.24253629; -0.38346055 0.3951012; 0.10675447 0.010848249; -0.31692457 -0.026318805; 0.47868532 -0.42820114; 0.41431135 0.012447005; -0.14703518 0.38260275; 0.6048018 -0.23120809; 0.34514105 -0.07317293; -0.8864168 0.27359483; -1.4898992 0.37373707; -0.006542587 0.20440874; 0.5671207 -0.06769265; 0.28469294 0.08822053; -0.3840529 0.14991032; 0.49310872 -0.041024636; 1.0778338 -0.2636088; 0.6775203 -0.0873828; -0.6997273 0.058863312; -0.1495981 0.04757312; 0.61004806 -0.03194547; -1.1729559 0.27103978; -0.82104003 0.20105867; 0.16132554 0.14542112; 1.6226861 -0.4160599; 0.75758433 -0.35831085; 1.1325083 -0.402214; -0.05889494 -0.21251523; -0.006180583 -0.15856011; -0.6403392 0.05619576; 0.22551152 0.11672373; 1.2285992 -0.3345474; -1.0379939 0.3745136; -0.34630528 -0.034626774; 1.5791305 -0.5047022], bias = Float32[0.37308487; 0.21701902; -1.2334192; 0.8260774; 0.9162292; -0.86153036; 0.05204648; 0.4532263; 0.46903986; 1.3075551; -0.8268877; -1.2375834; 0.4837869; -0.47050542; -0.55511963; 0.3067771; 0.77325886; -1.3715938; 0.735002; -0.42872128; -1.2354984; 0.7851556; 0.4700865; 0.07994914; 0.8489315; 1.1154802; 0.7277209; -0.256047; 1.5120263; 0.5170015; -0.077014506; 0.31515408; 1.2225983; -1.5816892; 0.85873365; 0.09612368; 0.7107153; 0.23812811; 0.32308543; -1.5175412; -1.1841085; -0.7375026; 1.2855228; -0.109926336; 0.7115491; -0.47920826; 0.39693955; -0.113968685; 0.16485624; -0.17176494; 0.12407618; 0.23504265; -0.7331125; -0.29360488; 1.1643862; 1.537784; 0.102274075; -1.2043313; 0.034792528; 0.6482183; -0.36256468; -1.111788; -1.12302; 0.28643304; 0.6912492; -0.21754004; 0.84342736; 0.71072686; -0.29240483; -1.2076862; -1.357724; -1.4453431; 0.6272033; -0.13705032; 0.4241586; -0.11141496; -1.5861309; 1.1541259; 0.2740176; -1.5909904]), layer_2 = (weight = Float32[0.33461708 0.40879425 -0.36993825 0.1701881 0.24477766 0.15559928 -0.22776048 0.04796663 0.0070090927 0.3818212 -0.13073488 -0.41275206 0.39544797 -0.039934143 -0.018612362 0.035455965 0.2580012 -0.242024 0.41469488 -0.09398249 -0.06028988 0.28465375 -0.11953456 0.0026025174 0.23301445 0.2870219 0.25547594 -0.23258425 0.2802397 0.3175477 -0.08895209 -0.027606698 0.45601678 -0.2742371 0.1903645 -0.09679374 -0.052920043 -0.27070498 0.033933416 -0.55571955 -0.4096789 -0.24693134 0.23060699 -0.1144066 0.032127492 -0.54904073 0.058329757 0.0060819816 -0.05770674 -0.3387165 -0.2967032 0.2505008 -0.30382854 -0.1356858 0.41809723 0.3376441 -0.09442626 -0.42520228 0.11439814 -0.022213994 -0.13559046 -0.48232228 -0.57054126 -0.09537159 -0.23761447 -0.05018299 0.1827251 -0.0667877 -0.21270967 -0.2115679 -0.34794614 -0.5310534 0.39442444 -0.15292922 -0.0048851646 -0.39370942 -0.36799452 0.2671857 -0.01971972 -0.2843546; 0.032274522 0.039591003 -0.3825252 0.27928075 0.17721282 -0.20794383 0.26482415 -0.3454737 0.19461551 0.39305493 -0.4563102 -0.33457145 0.2641643 -0.28616232 -0.27276167 0.49187604 0.48657298 -0.29366168 0.2285911 -0.46896392 -0.22322214 0.22012793 0.35467285 0.35845703 0.16574182 0.2735118 0.116315216 0.27316004 0.4749284 0.113258116 0.2921594 0.48114666 0.3470539 -0.33070964 0.32394665 0.29355732 0.49087638 0.32397652 0.28157052 -0.4050132 -0.24168347 -0.16814966 0.15511025 -0.18558836 0.31931323 -0.4202532 -0.29903382 0.4693086 0.36054444 0.2825776 0.17691112 0.3031081 -0.14213502 -0.29061437 -0.10131075 0.4008535 -0.18477783 0.04834919 -0.5603311 0.38998875 -0.4708652 -0.38876185 -0.38250098 0.45438465 -0.32824492 -0.48416623 0.5157436 0.4725154 0.17956159 -0.50764316 -0.40125033 -0.3051873 -0.16301297 0.2562499 0.4493927 -0.27303767 -0.2561733 0.3754286 0.23946829 -0.44456998; -0.24148637 -0.16795222 0.4607801 -0.3355346 -0.25774363 -0.015176845 0.29012147 -0.21237698 -0.0065202946 -0.22457142 0.14438339 0.41379273 -0.45049483 -0.15663512 -0.17677458 0.08917911 -0.4176995 0.24675998 -0.38009098 -0.2106186 0.15787788 -0.38807192 0.16590247 0.21646032 -0.57077676 -0.29943234 -0.43128443 0.26389003 -0.5972838 -0.23376124 0.015098338 0.006419941 -0.33853567 0.27817732 -0.4588638 -0.025573444 -0.07255597 -0.005079637 0.03341348 0.42135328 0.58953303 0.36155555 -0.36269936 0.06045511 -0.06166567 0.16784884 -0.32560107 0.23073606 -0.02403361 0.02427311 0.08750923 0.10311062 0.42297834 -0.21036011 -0.514128 -0.5030127 0.19012967 0.36945683 -0.09834681 -0.07063966 0.09116411 0.50236654 0.4296619 -0.0720239 -0.19884235 -0.10007975 -0.22188927 0.022894526 0.09979626 0.31267905 0.49040562 0.28298935 -0.10001983 0.17176116 -0.045907646 0.24422684 0.48706794 -0.4724676 0.13374212 0.6249627; 0.29367644 0.23053057 -0.16998759 0.449879 0.25998586 0.027147468 -0.06409859 -0.21435606 -0.2885 0.29986647 -0.13045391 -0.269464 0.40568376 -0.20368138 -0.12005806 -0.10270766 0.42399237 -0.37929714 0.30645072 -0.12945868 -0.18000078 0.4576558 -0.15483838 -0.21659131 0.04809207 0.20906068 0.471115 0.22358292 0.16596626 0.46599752 -0.124349646 -0.22374265 0.2474959 -0.32385382 0.16963907 -0.24859536 0.08962067 -0.11042872 0.17022389 -0.37802488 -0.44325605 -0.20776054 0.0065459316 -0.13330181 0.23819885 -0.21192233 0.075472906 0.33784112 -0.02404806 -0.23951097 0.20146711 0.20426202 -0.35193875 -0.39350095 0.19323456 0.22377196 -0.16139601 -0.29938653 0.0031855016 0.289096 -0.12231396 -0.22405347 -0.27044958 0.064689785 -0.28061432 0.14357212 -0.06623207 -0.1954775 -0.2122382 -0.30719832 -0.28975236 -0.071217366 0.18839268 0.10061283 -0.111003526 -0.38097683 0.093410775 0.24471965 0.022553625 -0.1955609; -0.048153035 0.031856976 0.23906457 -0.18929759 -0.009841381 -0.14396904 0.09813737 -0.32673827 0.13079235 -0.22376078 -0.04945383 0.1870933 -0.14060329 -0.20257291 -0.20394015 0.01971409 -0.14305183 0.4313679 -0.43218523 -0.26884028 0.30711457 -0.23704958 0.20714147 0.31437272 -0.5189917 -0.1819319 -0.15191823 0.24563493 -0.46264037 -0.28504542 0.16505383 0.24080339 -0.454224 0.7643068 0.035883367 0.31757653 0.048025828 0.34799913 0.03691814 0.46291846 0.36428943 -0.1462214 -0.591154 -0.19150732 0.2648118 0.0667382 -0.213536 0.37067753 0.1952312 0.2313561 0.3716392 0.10906565 0.27728727 -0.16933481 -0.4263679 -0.3652412 0.1103169 0.5203278 -0.11125016 0.18689626 -0.28156456 0.41764554 0.29827964 0.2330748 -0.4546563 -0.17693469 0.17369851 -0.12271715 0.23839259 0.31550923 0.40619695 0.5892376 -0.15148096 0.41509265 0.212732 0.016639419 0.21572426 -0.32489184 0.15077642 0.62432086; -0.036115035 -0.28458804 0.18495344 -0.18259765 -0.41078892 0.27461177 0.1294712 -0.064605184 0.15784022 -0.24422519 0.2647911 0.5151844 -0.48436943 0.120461755 0.20497957 0.072211914 -0.22240545 0.36903492 -0.15347579 -0.0018583358 0.4339104 -0.36735848 -0.17982626 0.09635181 -0.2361838 -0.5007165 -0.4244071 -0.08813404 -0.123172104 -0.10110672 0.06386467 0.09609763 -0.27519226 0.4253258 -0.46667755 0.19613594 -0.28774792 -0.11607527 -0.25617298 0.54573697 0.3635491 0.32812214 -0.45266822 0.11839287 -0.18931414 0.13599169 0.008363336 -0.19634897 0.11337961 0.33459473 -0.12172101 -0.3786094 0.22521809 0.21925016 -0.25822926 -0.14821334 -0.05335884 0.3388616 -0.22289643 -0.08963179 -0.053515553 0.2617658 0.3531535 -0.009884014 0.08596881 0.09309446 -0.048428055 0.09593992 0.1457005 0.23171681 0.18996385 0.15616249 -0.038278624 -0.15641052 -0.10496756 0.25343552 0.42899397 -0.4049709 -0.12824467 0.36801884; 0.24054505 0.089625366 -0.36320183 0.17057191 0.15906486 -0.31875834 0.0043747406 0.246997 0.13685212 0.47875452 -0.15312818 -0.56045586 0.22338563 -0.09935357 -0.19692671 -0.174563 0.15874347 -0.42736745 0.43772328 -0.096111305 -0.29488224 0.5211183 0.04525216 0.015687091 0.45270422 0.56541365 0.15075633 -0.08586343 0.5045801 0.23019399 -0.2523985 -0.10070981 0.28050604 -0.46600166 0.16684626 -0.103705615 0.3649541 -0.038764622 0.08405514 -0.25303492 -0.21081427 -0.3202844 0.56165075 -0.07883257 0.22994626 -0.38212985 0.10124075 -0.13623242 0.07338872 -0.2130802 -0.12674543 0.2608629 -0.22332297 -0.25587195 0.19817738 0.3080907 -0.056165032 -0.13495609 0.04564324 0.22969534 -0.22877108 -0.19964448 -0.4442337 0.02564152 -0.06321247 -0.12355311 0.21893975 0.33525553 -0.018024776 -0.66491616 -0.5466549 -0.30440906 0.17538033 -0.23632838 0.12979645 0.023962129 -0.5493937 0.393527 0.11415945 -0.61616576; 0.28477442 0.13231048 -0.3956383 0.29920095 0.24095291 -0.0053778533 -0.2717027 0.2062244 -0.2530767 0.18515939 -0.23191406 -0.54560477 0.4092828 0.049972393 -0.099284746 0.019556027 0.189197 -0.18764631 0.35769144 0.14685367 -0.06699311 0.32437938 0.024041642 0.015870722 0.30605057 0.29987842 0.4196245 -0.22519405 0.14221707 0.30173573 -0.35900542 -0.021812467 0.6299222 -0.467631 0.24050453 -0.027793022 0.17042221 -0.34555176 0.06061227 -0.4505182 -0.17415315 -0.38568446 0.43168935 -0.056520462 0.05164402 -0.1705001 0.029336868 -0.030244105 0.0032610302 -0.36357874 -0.28693333 -0.061574824 -0.14081836 0.15872909 0.44020572 0.3139284 -0.017886255 -0.42708677 0.08619214 0.108633436 -0.16077031 -0.46032256 -0.47868434 -0.14795114 -0.06706061 -0.04245086 -0.022358483 -0.049947858 -0.2345475 -0.3061241 -0.36968026 -0.19682916 0.15550607 0.003087511 0.052613515 -0.35091317 -0.20617855 0.19240521 0.06206283 -0.3821095; 0.028890861 -0.03625557 0.26114687 -0.19588642 -0.42114615 0.528907 -0.40293035 0.20066181 -0.45452043 -0.3107287 0.41076458 0.124909066 -0.13512415 0.18910788 0.18480092 -0.29384732 -0.17227514 0.28472883 0.04404032 0.23840117 0.19666867 0.29694325 -0.35446435 -0.40191802 -0.25396177 -0.28458744 0.21606532 -0.09105273 -0.19813268 0.15131901 -0.42611763 -0.5772237 -0.40524325 0.18303911 -0.13966143 -0.4917223 -0.21415348 -0.3096811 -0.207697 0.28661543 0.27021697 0.4718522 -0.23594448 0.21763723 -0.489714 -0.020477582 0.403677 -0.18867226 -0.3421048 -0.3534773 -0.28295523 -0.25198683 0.30347544 0.23727185 0.29695818 -0.46170145 -0.17778769 -0.49505183 0.45972517 -0.42308754 0.25512633 0.2444528 -0.016555674 -0.2779863 0.14073275 0.48444813 -0.18850185 -0.12754737 -0.0151145635 0.10410626 0.34977496 0.45141953 0.20091793 -0.36889908 -0.50660837 0.1598793 0.3733977 -0.488712 -0.37195763 0.29360268; 0.06779519 0.010259952 0.51378644 -0.32839262 -0.34638172 0.50935364 -0.27158394 -0.15871195 -0.1947988 -0.3373726 0.44402075 0.28781423 0.01686998 0.24381931 0.15486814 -0.3481941 -0.30140263 0.35227415 0.08073248 0.22581106 0.22929576 -0.1547024 -0.3896421 -0.43720186 -0.05464422 -0.575493 0.12886861 0.0026862926 -0.47886765 -0.26081184 0.0043796026 -0.4955135 -0.5028993 0.29304925 -0.1443823 -0.30581957 -0.27418706 -0.30177054 -0.2850587 0.3341473 0.6008719 0.28481978 -0.2830336 0.028848266 -0.4466957 0.15819141 -0.08910892 -0.07499038 -0.30239797 -0.12820947 -0.2761379 -0.053882003 0.09971828 0.2493402 -0.0097504575 -0.47383142 -0.08320839 0.085884154 0.49123225 -0.42024532 0.5263166 0.14726204 0.058854092 -0.3402593 0.06171838 0.37499595 -0.36735728 -0.29756975 0.0005468922 0.4832171 0.33178902 0.21414834 0.14311492 -0.17668724 -0.3403998 -0.04195328 0.5726623 -0.5781713 -0.43364418 0.2730859; 0.12880786 0.29522392 -0.2562987 0.36733446 0.10904668 -0.13839372 -0.055444695 0.049798146 0.1425636 0.15426101 -0.23080951 -0.25790465 0.5886965 -0.18018913 -0.18747245 0.1439605 0.460486 -0.22987463 0.30448425 -0.11996192 -0.28481728 0.5097976 -0.009006133 0.06979278 0.41374445 0.19726102 0.18186824 -0.26418966 0.31213427 0.24426271 -0.033711698 -0.008122582 0.6018059 -0.23673877 0.21064803 0.021396268 0.083463676 -0.06166815 0.07463428 -0.32707563 -0.39921182 -0.3914349 0.56845975 -0.23213725 0.112780355 -0.20409665 -0.059155587 -0.035408754 -0.012874074 -0.113444686 -0.14690697 0.18995102 -0.27834332 -0.3031972 0.39772034 0.4537729 -0.17259362 -0.34140766 0.13300283 0.13244115 -0.21869725 -0.2717875 -0.28779215 0.2348397 -0.06396554 -0.20766982 0.24355794 0.16958217 -0.25444013 -0.59180737 -0.3181682 -0.18909602 0.26103908 -0.09965679 -0.10413425 -0.24112277 -0.33556944 0.32354343 0.08109841 -0.22953542; 0.42063025 0.21498756 -0.4899378 0.28346068 0.23904143 -0.34126744 -0.001540922 0.035433453 0.08841906 0.17410369 -0.31592315 -0.36495128 0.34520248 -0.11128815 -0.40070662 0.026311714 0.4506319 -0.2632558 0.41229713 -0.1919225 -0.3154817 0.27992252 0.17983176 -0.028810434 0.3011493 0.16331562 0.4804012 -0.25742802 0.31248805 0.42269364 -0.051383138 0.042432215 0.32749802 -0.48378497 0.34356728 0.06490993 0.16158874 0.03016832 0.06640869 -0.48262283 -0.25811806 -0.4040287 0.34523883 -0.17370006 0.22163448 -0.23459935 0.006687311 0.32997644 0.097122796 -0.21449757 -0.13696077 0.09967865 -0.1771323 -0.31733224 0.3929034 0.31913313 -0.37148502 -0.32791105 0.06443982 0.07748938 -0.2168896 -0.37135592 -0.60585815 0.09813786 -0.017855486 -0.30616924 0.17766748 0.30361938 -0.09502518 -0.304009 -0.1740269 -0.4104346 0.15542781 -0.05959146 0.23305105 -0.40818188 -0.47776031 0.39442056 0.21047649 -0.35807607; 0.3517906 0.25669894 0.24009992 -0.2933264 -0.21832982 0.3843834 -0.5064912 0.22662121 -0.19454575 -0.4579296 0.2608241 -0.15910628 0.100053206 0.44869995 0.29631284 -0.4340043 -0.24020028 0.17609736 0.24758951 0.4445661 0.41913548 0.3379902 -0.26842496 -0.25154746 -0.042362425 -0.13913523 0.16452302 -0.12966998 -0.25607026 0.0020253055 -0.1836872 -0.45134377 0.07798874 0.39965278 -0.4141949 -0.43168882 -0.3819784 -0.32980397 -0.4891655 -0.09420176 0.13506591 0.47339457 0.0594885 0.21121407 -0.2907123 0.09145878 0.3771282 -0.43758652 -0.55738336 -0.3838497 -0.332151 -0.49011207 0.02113285 0.2545777 0.4116036 -0.40879798 0.06909613 -0.27261156 0.38555276 -0.42875293 0.5133954 0.08548068 -0.20132113 -0.3043841 0.118788876 0.46820524 -0.26138166 -0.5199695 -0.14607224 0.18077543 0.3493828 0.22491693 0.45312247 -0.17802423 -0.43988556 -0.09023301 0.43409747 -0.31808645 -0.23620602 0.08794565; -0.2769421 -0.16674009 0.2901834 -0.1432678 -0.473246 0.4581039 -0.11552601 -0.17641832 -0.18066174 -0.25777674 0.42257336 0.57632184 -0.4378541 0.17835967 0.36631238 -0.25849903 -0.33915383 0.5555749 -0.44479048 0.039486825 0.5590794 -0.19098108 -0.25522766 -0.0007131875 -0.23174982 -0.550227 -0.096336946 0.33953923 -0.61303276 -0.26514158 0.062400557 0.16017672 -0.3731647 0.62764984 -0.31297928 0.13681635 -0.2901096 -0.121208884 -0.36211738 0.510611 0.5464224 0.40652588 -0.5931786 0.024592936 -0.22046652 -0.01872312 -0.05799608 0.08082969 0.018969342 0.02703605 0.29227605 -0.19298494 0.093478255 0.16663049 -0.2165682 -0.57485217 0.014979784 0.41007236 -0.15550932 -0.35630858 0.2340925 0.38467216 0.44292867 -0.33115035 -0.11217132 0.19208382 -0.4705107 -0.5332692 0.1062841 0.65658116 0.53078824 0.3595533 -0.022678878 0.12727973 -0.36216703 0.055228785 0.6161264 -0.36969426 -0.14443845 0.37922415; -0.021610014 0.016673876 -0.4225761 0.29580176 0.2141387 -0.27916068 0.3225807 -0.0011041302 0.32085586 0.15375914 -0.40543517 -0.21196538 0.21265619 -0.25444555 -0.14338411 0.33993432 0.11516647 -0.22921893 0.3579646 -0.25077787 -0.31999204 0.36661053 0.42542252 0.46506697 0.3358001 0.50267434 0.13019738 0.05310503 0.2954739 0.3233826 0.21083264 0.27304158 0.36270234 -0.32630906 0.31282765 0.3208243 0.4152339 0.3559298 0.25645557 -0.5608399 -0.31029013 -0.4911079 0.43613425 -0.34310374 0.22754456 -0.110150605 -0.1596977 0.011073472 0.41499177 0.045322273 0.10802693 0.16284797 -0.42913404 -0.33560073 0.275799 0.37432468 0.09318987 -0.036672063 -0.3456104 0.15899205 -0.34666735 -0.21663705 -0.18163009 0.3230165 -0.08160551 -0.1674909 0.25058332 0.4422996 0.044163976 -0.3243816 -0.25636303 -0.15314285 -0.094940916 0.16555524 0.4379705 -0.096102856 -0.5079026 0.15736978 0.3757731 -0.37732163; -0.34167174 -0.09905121 0.2850254 -0.14294039 -0.3655668 0.04450197 0.312301 -0.1348002 0.11477991 -0.4868398 -0.008055666 0.5697432 -0.62002397 -0.1326237 0.19194102 0.077921964 -0.4910236 0.17373501 -0.26236454 0.017479185 0.23662943 -0.54969233 0.22959226 -0.059478406 -0.5083525 -0.1744887 -0.30753362 0.040302187 -0.3645376 -0.26222134 0.34448794 0.085674465 -0.43931088 0.45414105 -0.33540115 0.26430216 -0.09125381 0.14369254 0.15637703 0.43082115 0.23188403 0.2615055 -0.4278636 0.32565033 -0.16036905 0.4477835 -0.09448788 0.18401127 0.28987342 0.18612663 0.33030462 -0.12720026 0.47379377 0.19795156 -0.47875658 -0.46117312 0.3570645 0.34453684 -0.02181423 -0.1133155 -0.109554 0.39463603 0.2048155 0.1604388 -0.05802778 -0.08043328 -0.10423713 0.111918725 0.24164072 0.37937447 0.47150722 0.41742155 -0.38979316 -0.02071532 0.15381339 0.11333236 0.121096864 -0.36593294 -0.08897123 0.27033854; -0.17114903 -0.45415393 0.28419656 -0.24160163 -0.10104561 -0.07312658 0.10831005 -0.3594395 0.24736877 -0.14192519 0.09181248 0.57266706 -0.69307476 0.0444471 0.046382084 0.08279767 -0.21637687 0.2017215 -0.30102178 0.06452613 0.059338752 -0.23574318 0.036623072 0.0014056817 -0.30195907 -0.24054402 -0.25856274 0.10683754 -0.14296937 -0.4359826 0.12371612 0.2501642 -0.5521749 0.1012117 -0.08204464 0.0038407713 0.14509308 0.26074374 -0.03755569 0.46349734 0.36697912 0.4309217 -0.50719494 0.45780495 0.03396893 0.37472156 -0.047224365 0.07604961 -0.032876343 0.17919475 0.18427144 -0.19254449 0.17288522 -0.030637374 -0.25867188 -0.2878607 0.3971481 0.2077246 -0.18702061 0.06764258 0.1283216 0.24298087 0.5519697 -0.07863741 0.025000574 -0.036112905 0.0033875385 -0.071274966 0.18783413 0.47543964 0.2390897 0.3650592 -0.29111335 0.16176899 -0.07812388 0.4219564 0.2600146 -0.3335603 0.16145617 0.27434817; 0.19939092 0.04663603 0.16753332 -0.397006 -0.19886608 0.3547967 -0.45555484 0.037276417 -0.20058614 -0.30632663 0.42307037 -0.15526277 0.20654675 0.32690623 0.22020665 -0.36415732 -0.03924912 0.10690412 0.14749381 0.33408245 0.20600612 0.4079488 -0.37611547 -0.27583867 -0.054388955 -0.35508686 0.27144122 -0.40974772 -0.15798299 0.18668129 -0.480106 -0.26159248 -0.30000877 0.33566156 -0.4522652 -0.33744 -0.24299164 -0.44582158 -0.3602049 -0.052322548 0.18831691 0.31829783 -0.2542513 0.28747502 -0.3204051 -0.10005292 0.20856236 -0.33351046 -0.46165276 -0.4810921 -0.34307513 -0.45205992 0.04620487 0.3652578 0.35255218 -0.10979099 -0.22319445 -0.5973685 0.34520954 -0.47464752 0.3812094 0.09776196 -0.25326556 -0.27799976 0.043079548 0.3944492 -0.5074887 -0.1936647 -0.24054925 0.22624838 0.35341218 0.13007006 0.42948127 -0.27083847 -0.22348715 -0.003413899 0.3383339 -0.19061688 -0.4095645 0.05670742; -0.3706594 -0.32051235 0.42921668 -0.42951664 -0.32455206 0.09839507 0.1403915 -0.34461102 -0.09956774 -0.18734363 0.15222119 0.32323202 -0.48294953 0.17149276 -0.11617784 0.054922707 -0.327029 0.33976883 -0.2295172 0.08402082 0.3759331 -0.16284314 0.13443027 0.12769863 -0.30644464 -0.3587649 -0.41195908 0.36463764 -0.5292288 -0.35000572 0.117765345 -0.015294771 -0.6331106 0.599465 -0.1450672 0.1470898 0.15136838 0.13507833 0.098553486 0.5486406 0.47382352 0.18022797 -0.4947659 0.0073614023 -0.19717643 0.462751 -0.021116585 0.23276043 0.052793317 0.106576376 0.03015852 0.13821639 0.2746949 0.0100743165 -0.1952527 -0.4117627 -0.0030653968 0.15785044 -0.24258006 -0.20459753 -0.10210266 0.24606685 0.3586948 0.16899158 -0.3327043 -0.26204067 -0.075868145 0.024570847 0.35009232 0.545365 0.41534874 0.65545434 -0.31957453 0.1815132 -0.107415095 0.15309419 0.5104588 -0.29826483 -0.07322285 0.44239828; 0.13145362 0.28092492 -0.1830806 0.23215392 0.21954069 0.12960435 -0.022252714 0.2548448 -0.01927445 0.24313895 0.2829596 -0.5328914 0.43989503 -0.037517298 -0.030338453 -0.1211696 0.24276437 -0.52546483 0.24247994 0.14118764 -0.24498555 0.44004062 -0.3164047 -0.0923632 0.45442 0.19546494 0.3022808 -0.35761616 0.24342357 0.48826036 -0.40154418 -0.029494738 0.5368837 -0.42411697 0.287171 -0.24297029 -0.15853146 -0.26164025 -0.25268328 -0.535987 -0.2691916 -0.07286596 0.40356857 -0.1186602 -0.094064616 -0.2785657 0.19643626 0.03118572 -0.14563844 -0.009356403 -0.15825671 -0.048962537 -0.46990874 -0.000780249 0.47343385 0.32274622 -0.33822465 -0.5206743 0.107945405 0.019125242 -0.020432489 -0.18905176 -0.4712686 -0.28108615 -0.020566696 0.04134625 0.13742287 0.04076574 -0.2467507 -0.50031495 -0.2897041 -0.5286717 0.49944982 -0.13774575 -0.34298557 -0.37897193 -0.120136335 0.16343908 -0.14288968 -0.24523522; 0.45961174 0.22755873 -0.1720512 -0.1466789 -0.18861468 0.38164225 -0.2863769 0.14470828 -0.42994353 -0.058505967 0.15442443 -0.21137357 0.3749356 0.34812668 0.1154817 -0.18731415 -0.012519191 -0.067816004 0.35748062 0.32962677 0.27287892 0.5855523 -0.46340016 -0.39383852 0.24695806 -0.10296629 0.014806361 -0.45957246 -0.14543895 0.07363939 -0.21894291 -0.43089488 0.37635204 0.1126895 -0.47579855 -0.31941086 -0.4552753 -0.4909982 -0.37809825 -0.071183816 0.197238 0.18003783 0.20516628 0.0037022242 -0.46351022 -0.19621955 0.17074797 -0.27383092 -0.21231467 -0.20720272 -0.32937488 -0.50511956 -0.12071678 0.47040638 0.3657822 -0.10814578 -0.0046533286 -0.37754276 0.5066575 -0.3422037 0.4670523 0.018987179 -0.33198404 -0.2837175 0.16921428 0.3740689 -0.074104294 -0.40487248 -0.21353541 -0.27787617 0.4791036 -0.09805639 0.18975918 -0.06923194 -0.439956 -0.14004095 0.2681103 -0.35783076 -0.22555749 -0.028553417; -0.079819284 -0.21599719 0.44498563 -0.29736105 -0.27052924 0.19041148 -0.077735856 0.099690594 -0.42833716 -0.17109773 0.2996602 0.41896632 -0.40950614 0.14889866 0.4452883 -0.26189044 -0.27393764 0.4170042 -0.1538269 0.11123364 0.39953285 -0.2643313 -0.2233403 -0.44236973 -0.44877478 -0.47021708 -0.4159402 -0.048008073 -0.49605215 -0.40890428 0.085447274 -0.3232782 -0.21180002 0.41345078 -0.14987738 -0.13066588 -0.42543605 -0.2872445 -0.14231494 0.532158 0.17366414 0.13875897 -0.49998662 0.18886843 -0.2968572 0.42401952 0.071592025 -0.027577247 -0.093791135 0.18305734 -0.14296415 -0.15017827 0.34662005 0.24348457 -0.26624072 -0.4301035 -0.073565334 0.22028713 0.24431469 -0.12972215 0.3142053 0.3186007 0.18272796 -0.4694951 0.061500806 0.29716522 -0.42887062 -0.1491387 -0.00012642285 0.5034299 0.46111116 0.5018349 -0.28700814 -0.25047383 -0.2924464 0.105634175 0.38584217 -0.38387775 -0.21105155 0.33542207; -0.31811017 -0.33332816 0.40325147 -0.1446708 -0.17464565 -0.043622177 0.2432009 -0.19710506 0.14665799 -0.12820648 -0.27091998 0.2898483 -0.59595037 -0.13207826 -0.03619279 0.34058598 -0.3939249 0.40216255 -0.28967533 -0.1503567 0.216329 -0.2876133 0.3162722 0.07175571 -0.33865228 -0.3205839 -0.39812785 0.32094878 -0.12197532 -0.24698626 0.23751763 0.030356193 -0.6166227 0.019305313 -0.18791845 0.30997851 0.27195832 0.17082146 -0.03450238 0.3568245 0.3799662 0.10457238 -0.31557196 0.3361435 0.23313954 0.24672548 -0.1711825 -0.05067303 0.37349284 0.14250544 0.35665157 0.0717386 0.38907406 -0.07727735 -0.419656 -0.26149306 0.19871226 0.27206534 -0.24549958 -0.04620878 -0.058938105 0.298022 0.3676251 0.3002954 -0.026808433 -0.21383496 -0.04923568 -0.052322816 0.12404023 0.26945317 0.40450165 0.4224365 -0.44803998 0.26618445 0.27066684 0.1620104 0.15462169 -0.45994514 0.1809883 0.2023659; 0.23106124 0.4776868 -0.23833707 0.15914464 0.42805398 -0.27188 0.16001089 -0.16405647 0.3366311 0.5126421 -0.4505323 -0.2646202 0.41142213 -0.472954 -0.15895383 0.46720442 0.2356089 -0.18839884 0.338714 -0.14078045 -0.4498894 0.19936979 0.19395882 0.11659191 0.1758328 0.43953916 0.54541516 0.16668242 0.35989475 0.20925963 -0.07268803 0.44609168 0.42232653 -0.15979627 0.17915833 0.15802288 0.36254156 0.09758422 0.28306088 -0.31012478 -0.18513222 -0.35965827 0.34437773 -0.38240543 0.34681582 -0.23215285 -0.14896509 0.12441489 0.119950645 0.1200451 0.33300936 0.13299222 -0.23598482 -0.135614 0.06403995 0.4062814 -0.2422571 -0.3520538 -0.15375897 0.13050932 -0.13402328 -0.14720145 -0.22751614 0.34121677 -0.243992 -0.29003254 0.12721452 0.23038203 -0.21969691 -0.5024712 -0.47579086 -0.4384474 0.20248668 0.25687283 0.19671237 -0.30915305 -0.4199958 0.24056724 0.44824705 -0.24418516; -0.21126688 -0.21715997 -0.13715841 0.2762579 0.43088552 -0.52623785 0.5298602 -0.2928426 0.2158133 0.53065753 -0.46832606 0.103283584 -0.33964562 -0.5306146 -0.3460164 0.39183918 -0.11648888 -0.2730599 -0.41763347 -0.5284039 -0.23042114 -0.21971509 0.40856758 0.5436126 -0.2216685 0.56529903 -0.03657496 0.15087263 0.19812442 -0.049076665 0.36893958 0.27189472 0.07122341 -0.23479474 0.27067587 0.36882728 0.4193566 0.3847501 0.37081826 0.005473249 -0.5077359 -0.47245625 0.25664067 -0.17456165 0.65143013 -0.09393761 -0.07973153 0.11352934 0.54276085 0.50575376 0.4093345 0.4095424 0.006843961 -0.3403713 -0.117888905 0.45507106 0.20802854 0.30271244 -0.49926075 0.4902345 -0.3732124 0.09592188 0.19267747 0.4921318 0.050673064 -0.44160903 0.17569074 0.13112423 0.02016846 0.08243354 -0.32830656 -0.29398867 -0.47591177 0.31009343 0.26534778 0.15775116 -0.16824219 0.46968332 0.26807216 -0.017514467; 0.25713897 0.10958351 -0.2769891 0.29396906 0.45946962 -0.0016967355 -0.15173994 0.3526857 -0.07240512 0.5376204 -0.2738579 -0.36830598 0.13237792 0.013005196 -0.226513 -0.13828668 0.25399682 -0.31758136 0.320593 0.07273527 -0.64782757 0.36269397 -0.2217713 -0.12657578 0.54144967 0.27278945 0.33856916 -0.46884903 0.53263825 0.24329013 0.0077567054 -0.2218383 0.6316866 -0.59659094 0.16768456 0.08863442 0.2642229 -0.03043156 -0.2898469 -0.6949162 -0.5975169 -0.4014791 0.48272094 0.13488623 0.07133522 -0.14971723 0.34449488 -0.33553433 -0.16164474 -0.2611768 -0.12966911 -0.2724275 -0.3389954 0.26608777 0.5244783 0.7614954 0.22269084 -0.54481876 -0.005336871 0.10783517 0.13529016 -0.43316963 -0.23663838 -0.117741 0.3616307 0.022884673 0.17528783 0.23779675 0.053022645 -0.7367376 -0.2771492 -0.5980244 0.16056369 -0.17230015 0.040283397 0.19189933 -0.70022136 0.5665397 -0.19375555 -0.72083795; 0.35349247 0.1499285 0.30890277 -0.49955162 -0.54634714 0.42797956 -0.26653785 0.40682527 -0.23585226 -0.37946647 0.36508873 -0.19653273 0.26825738 0.43930086 0.25941426 -0.44474408 -0.11353877 0.34197333 0.27805725 0.3173191 0.33433646 0.3616007 -0.3019547 -0.29643598 0.041884046 -0.3870428 0.1755827 -0.16970643 -0.3261024 0.107013896 -0.21677549 -0.19142592 0.02944793 0.25193784 -0.3034545 -0.26032838 -0.2736446 -0.48739222 -0.30557036 -0.19190644 0.047082584 0.38060594 0.18828782 0.020250292 -0.28301683 -0.03858607 0.39877284 -0.31766713 -0.41280273 -0.2542058 -0.22415471 -0.4062567 0.012747292 0.2427081 0.3365162 -0.15813656 0.054043975 -0.45798486 0.2561246 -0.24959071 0.23141168 -0.079061635 -0.27856547 -0.43019423 0.3520802 0.31357792 -0.25685388 -0.1641246 -0.12263835 -0.12566847 0.3165221 0.0349176 0.33788595 -0.22631952 -0.2818591 0.044729643 0.3439243 -0.14014293 -0.48203355 0.18562499; -0.071316786 -0.38392356 0.19554026 -0.28727272 -0.29036093 0.11795748 -0.4433892 0.40815714 -0.25663283 -0.3109275 0.21345755 -0.059877794 -0.037197016 0.27112922 0.42016718 -0.09705567 -0.46984893 0.3447395 -0.29462314 0.24567951 0.12047373 0.17406 -0.5359503 -0.269003 -0.25571954 -0.073571265 0.05720041 -0.3548869 -0.08278375 -0.049781624 -0.009661228 -0.3731403 0.025721975 0.2910273 -0.21302295 -0.30154172 -0.049179237 -0.18295391 -0.5095063 0.081115484 0.36190665 0.29732677 -0.15578489 0.58059305 -0.25433996 0.12731682 0.4912377 -0.34842637 -0.34785995 -0.19083065 -0.34342206 -0.387176 0.3943025 0.25790066 0.55276376 -0.19981141 0.32601917 -0.31194595 0.462185 -0.39786106 0.42670363 0.2915444 0.13207357 -0.47974795 0.5267524 0.17445493 -0.012397786 -0.34382275 -0.0761437 -0.07953071 0.23449957 0.19076465 0.17500037 -0.57201344 -0.1462106 0.38703063 0.27862138 -0.16755687 -0.53962135 -0.10761982; 0.08211252 0.1793404 -0.39208183 0.12767544 -0.020652344 -0.020771159 -0.27598542 0.41014016 -0.1807245 0.49124038 0.16558751 -0.5328223 0.20852646 0.15154214 0.26606992 -0.14161895 0.2698176 -0.4245322 0.43468323 0.2824286 -0.15474555 0.15413004 -0.0664029 -0.075394586 0.62146276 0.35472503 0.43729287 -0.07334109 0.54541636 0.27166066 -0.41573974 -0.20555556 0.4177924 -0.7155706 0.16119511 -0.025200818 -0.24736191 -0.34297165 0.01594411 -0.54919636 -0.31238574 0.042466916 0.31500646 0.2299429 -0.07486214 -0.14056015 0.34273982 -0.3726785 -0.25646904 -0.12090352 -0.08501057 -0.09668679 -0.31359997 0.22508892 0.24705137 0.47640374 -0.04352676 -0.29261562 0.13454969 -0.0796684 0.30936682 -0.29930264 -0.4453819 -0.23910075 0.2720054 0.07413961 -0.109588094 0.041289885 -0.14172997 -0.2740008 -0.53393024 -0.44143334 0.0857891 -0.28286183 -0.17133404 -0.116048545 -0.43483996 0.5434337 0.0031122356 -0.5257087; 0.12473203 0.097543955 -0.30796278 0.3184735 0.35669062 -0.026918387 0.028881859 0.37084538 -0.24947155 0.50800765 -0.10981003 -0.41341126 0.30937937 0.001555455 0.20540524 -0.15438479 0.20627703 -0.5502657 0.34996498 -0.04321823 -0.37990645 0.33290905 -0.1036774 -0.3505718 0.48202026 0.5686049 0.3992602 -0.13377075 0.46600187 0.19595693 -0.42639434 -0.022002103 0.57448834 -0.672765 0.25386304 -0.31842616 -0.10625203 -0.041032963 -0.074338004 -0.6106666 -0.5748907 -0.28089264 0.27046388 -0.03233442 -0.08354045 -0.14895639 0.3420862 -0.09547375 -0.07459844 -0.09586276 -0.1291597 0.0330517 -0.42099908 0.21395485 0.438372 0.45746306 0.024176033 -0.5443231 0.326586 0.20356499 0.06601763 -0.37427545 -0.27408165 -0.24913037 0.4145838 0.2956484 -0.011261912 -0.09226679 -0.15561728 -0.4349286 -0.45669335 -0.5739068 0.24758878 -0.23399995 -0.22737566 -0.025144551 -0.46147472 0.3701839 -0.23262176 -0.5308962; -0.13342138 -0.16894354 0.21991481 0.28449604 0.17434646 -0.34427288 0.39875638 -0.06969186 0.43499336 0.121309355 -0.19284904 0.42545995 -0.24465342 -0.36429483 -0.43170947 0.15932903 -0.13502048 0.07987033 -0.21173999 -0.36596698 0.009852517 -0.5212825 0.19372508 0.42336097 -0.31842673 0.13501142 -0.29961032 0.13654268 0.1674391 -0.33968124 0.22051536 0.4569647 -0.35787037 0.15774156 0.41982582 0.3704001 0.10206485 0.26691064 0.48494208 0.45680848 -0.12194838 -0.25273603 0.04244327 0.08255354 0.37922212 0.009890975 -0.06745223 0.27256733 0.41809326 0.29593778 0.43867263 0.2573122 0.11154641 -0.18049707 -0.48637095 -0.2122305 -0.033570573 0.23030409 -0.15241453 0.49736512 -0.4902977 0.26353052 0.30043146 0.15399365 -0.10234895 -0.27415112 0.119839475 0.088351965 0.2630977 0.20452355 0.08776162 0.39742485 -0.4786092 0.44041073 0.447736 0.15755554 -0.18675596 -0.032169476 0.1940109 -0.03417863; 0.30195248 0.07797991 -0.4549593 0.30505294 0.22883858 0.30392522 -0.13224551 0.20000972 -0.062472943 0.2600879 0.2567801 -0.27732572 0.5152713 0.19143413 0.25807253 -0.1495013 0.1813609 -0.44791165 0.5942886 -0.015483888 0.0031635077 0.39680952 -0.17427142 -0.31038308 0.32185474 0.12658826 0.3925371 -0.18220578 0.08154137 0.4990153 -0.3705299 -0.08413614 0.2918204 -0.0653694 -0.11424659 -0.28512514 -0.22199932 -0.2956424 -0.3440575 -0.5328471 -0.15416838 0.005463016 0.29880595 -0.18862163 -0.22080554 -0.24488378 0.21638207 -0.23371737 -0.37857205 -0.46070728 0.025228187 -0.30189013 -0.37716722 0.28471836 0.3524571 0.022170978 -0.2268253 -0.49337834 0.099221945 -0.14345005 0.10983858 -0.3208943 -0.45693618 -0.12899438 -0.08517978 0.3015581 -0.3431492 -0.15543021 -0.27713764 -0.24546413 -0.36324978 -0.4146263 0.13644305 -0.15326044 -0.10211973 -0.1295065 0.20914029 0.12311449 0.030345175 -0.2795394; 0.09655711 -0.06609458 0.22725265 -0.44665235 -0.30101135 0.5406219 -0.22444715 -0.16809472 -0.37724435 -0.43022797 0.4037974 0.28854707 -0.0066199615 0.34260714 0.38012514 -0.12258125 -0.34437248 0.46840203 -0.08147857 0.40302092 0.551585 -0.23116079 -0.3747405 -0.4531439 -0.09475398 -0.57601947 -0.04274049 0.06890124 -0.5538821 -0.11315028 -0.22290173 -0.37665 -0.54843813 0.30753767 -0.35110506 -0.16192238 -0.31138313 -0.32810727 -0.37681773 0.473804 0.2572416 0.26113942 -0.61100334 0.18171138 -0.18428779 0.2868858 0.11894248 0.09421864 -0.15052772 0.24325453 0.04359438 -0.15649359 0.10915269 0.12584111 -0.17411116 -0.36245146 -0.18304847 0.27941537 0.3804275 -0.19079906 0.29141018 0.3537784 0.14444125 -0.3841716 -0.1539103 0.19491999 -0.43188718 -0.41363564 -0.08514114 0.5200049 0.5099712 0.28179237 0.1007181 -0.15791936 -0.23963702 0.1346247 0.55351454 -0.53684676 -0.19147633 0.5723389; -0.119731195 -0.42567325 -0.10287119 0.08530253 0.31053618 -0.4727563 0.28836206 -0.38388774 0.17927478 0.3309836 -0.4957613 0.16182436 -0.52102244 -0.32621148 -0.33411014 0.36127228 0.0636237 -0.11560932 -0.66123265 -0.5572359 -0.35070813 -0.37730947 0.47734228 0.16623285 -0.50279444 0.35580236 -0.29391947 0.41519833 0.13667366 -0.32985866 0.49840832 0.5065337 -0.22684546 -0.3341076 0.44520882 0.24792077 0.40697423 0.28980723 0.34985092 0.07630626 -0.21824624 -0.22037143 -0.13591263 -0.060692023 0.6928006 0.047029 -0.326071 0.17797172 0.15676416 0.23122947 0.55523056 0.48635575 0.08580944 -0.4062435 -0.3776931 0.070056535 0.35500816 0.30345345 -0.17602704 0.36386472 -0.27629682 -0.046318583 0.46100223 0.17311192 -0.06254142 -0.20051327 0.15143049 0.4012725 0.27394968 0.2714655 -0.56460625 0.09840081 -0.30747706 0.11020205 0.25334528 0.0059559485 -0.32899612 0.3803901 0.29154387 -0.081163816; -0.17944379 -0.32579878 0.46107262 -0.3968029 -0.3404658 -0.29417443 0.06621424 -0.3444388 0.04132763 -0.20213962 -0.10757837 0.43960443 -0.6732549 0.07170924 -0.2552235 0.22896862 -0.16028544 0.44821283 -0.28191862 -0.19935395 -0.047357008 -0.43089947 0.06621305 0.367931 -0.4909234 -0.18297154 -0.41598555 0.3180368 -0.06421824 -0.26178354 0.36541736 0.12349575 -0.3349663 0.07227851 -0.07279765 0.06437324 0.0025313676 0.25847456 -0.0015559483 0.28852504 0.36541772 0.044071507 -0.29844293 0.19459648 -0.12765726 0.3738863 -0.1993738 -0.032323774 0.3707975 0.10179859 0.24597359 0.18584742 0.15154693 -0.28500387 -0.36671305 -0.35211465 0.2092034 0.3987643 -0.06416995 0.06171888 -0.22754245 0.24123184 0.22540784 0.31620282 0.08548733 -0.18462892 -0.019584252 0.11577229 0.3951483 0.22670633 0.33426917 0.4204653 -0.21383825 0.1826268 -0.017067263 0.2794051 0.22593737 -0.31764674 -0.018169295 0.28762946; -0.46190095 -0.2780568 0.16902687 -0.4593652 -0.35375983 -0.10994988 0.016375622 -0.008187147 0.051246334 -0.17449024 0.104444124 0.49451765 -0.34241384 -0.1370933 -0.081303574 0.28617144 -0.26725432 0.26547626 -0.42728838 -0.07768056 0.054654416 -0.36530071 -0.03638085 0.24059361 -0.54042125 -0.24782999 -0.25152674 0.059638485 -0.46879607 -0.4418545 0.043442503 0.20864418 -0.5619506 0.17880838 -0.18396768 0.2964617 0.05491509 0.27687475 -0.24803929 0.371873 0.16154996 0.11456868 -0.3361153 0.2525811 -0.10569821 0.5195514 0.030854901 0.014598152 0.08703956 0.06626375 0.08572766 -0.15169306 0.24243586 -0.051247634 -0.03899548 -0.21186385 0.12952237 0.44211066 -0.016295407 -0.04312713 -0.090857506 0.2993714 0.52905583 -0.08680891 0.2898224 -0.06347106 -0.2898359 -0.07695186 0.1749787 0.4010015 0.30078888 0.2924038 -0.10162762 0.23402707 0.16102757 0.26810172 0.0729072 -0.36008134 0.15083452 0.35794562; -0.21422264 -0.07852281 0.18124989 0.29354757 0.31145203 -0.3442433 0.17759216 -0.29844806 0.17546211 0.20613311 -0.31996328 0.17114094 -0.4399357 -0.53384644 -0.4550774 0.4281421 -0.1205217 -0.20977348 -0.30961397 -0.2540758 -0.32938102 -0.4533094 0.3307869 0.27492574 -0.23447399 0.1902079 -0.08448133 0.4486131 0.09519289 -0.37188485 0.16174316 0.20126048 -0.2563107 -0.29655263 0.17901136 0.34382078 0.36631075 0.55196434 0.60452354 0.11455868 -0.1841615 -0.34863812 -0.09940352 -0.09567393 0.33401328 0.22455806 -0.177265 0.11077112 0.5222894 0.3903565 0.28629068 0.26520205 -0.13563715 -0.5227387 -0.3629878 0.060468365 0.027293012 0.2583809 -0.2911782 0.47490314 -0.30825457 -0.07765022 0.06675012 0.20917289 -0.11937308 -0.38050607 0.19455719 0.31583092 0.4169841 0.01366134 -0.10386959 -0.104088455 -0.15640077 0.2625116 0.40198243 0.19597913 -0.3492548 0.09936151 0.18821917 0.1612557; 0.05273545 -0.13609998 0.5446253 -0.28052682 -0.45764667 0.533609 -0.06278255 -0.28080612 -0.31244442 -0.47829017 0.56649435 0.60268235 -0.20016178 0.33060333 0.49952596 -0.4274015 -0.24171363 0.5621677 -0.20535518 0.3405049 0.23573972 -0.2592685 -0.28668746 -0.17032714 -0.42080504 -0.49841958 -0.06651005 -0.05230656 -0.35344714 -0.12119436 -0.095586956 -0.07502706 -0.39759955 0.31795698 -0.21144219 -0.2469866 -0.48334298 -0.12635584 -0.43362093 0.3536872 0.2332274 0.46828425 -0.5343019 0.11480206 -0.3365728 0.14890495 -0.21382955 -0.17200293 -0.12471517 0.26526096 -0.07009418 -0.38638458 0.20256495 0.154436 -0.46139255 -0.60348207 -0.081513934 0.27841595 0.2903245 -0.25987136 0.36214787 0.4944997 0.28215092 -0.24652202 -0.05125959 0.27045122 -0.5937486 -0.44471937 0.17884174 0.38007563 0.261824 0.47791284 -0.22723004 -0.00876924 -0.33255997 0.12418591 0.53218293 -0.2939691 -0.09938317 0.385932; 0.1173731 -0.12570937 0.31649262 -0.22070734 -0.22535264 0.44045684 -0.29247043 0.24913435 -0.4647437 -0.1786373 0.27659166 -0.09810207 0.33176816 0.45618507 0.43180647 -0.43903142 -0.05927274 -0.037224155 0.19333224 0.54068923 0.42237133 0.368119 -0.19036824 -0.20165047 -0.1991204 -0.10641476 0.15309699 -0.40243524 -0.2742799 -0.025664033 -0.29744932 -0.24518798 0.17092006 0.16104679 -0.22101235 -0.44928104 -0.15789771 -0.22870648 -0.3829463 -0.08213528 0.19243257 0.30190095 0.29243255 0.286404 -0.31741685 -0.21143104 0.40885484 -0.55363226 -0.19916783 -0.15280274 -0.35779357 -0.5878401 0.08298161 0.28374723 0.4734888 -0.04899951 -0.021093104 -0.24869929 0.2011848 -0.5350242 0.49803308 -0.09718847 -0.14119335 -0.17492355 0.20290491 0.34089524 -0.3508471 -0.20184016 -0.22692104 -0.35801226 0.27927065 0.053321883 0.20317045 -0.26860985 -0.35346907 0.010751236 0.44360447 -0.31851944 -0.25030643 -0.060588617; 0.26626092 0.20372495 -0.41826344 0.24141356 0.2358029 -0.19465676 0.32565972 0.13377889 0.4027577 0.24039729 -0.37751138 -0.25205088 0.44611305 -0.3418773 -0.18617254 0.42847428 0.3713604 -0.18510148 0.50550437 -0.23382515 -0.4299788 0.30323023 0.11296898 0.4300684 0.3927289 0.31486177 0.33771655 0.079519585 0.3608242 0.17499857 0.04086419 0.35043648 0.49670202 -0.44826993 0.15742205 0.39572582 0.5223071 0.3819218 0.33804002 -0.49482998 -0.24092843 -0.15647148 0.5461505 -0.40227264 0.2370294 -0.27787548 -0.033531398 0.008510691 0.32592353 0.040210154 0.14957182 0.27446806 -0.109917365 -0.21056737 0.47034007 0.47268972 0.06963794 -0.47960213 -0.006976438 0.30976495 -0.3730358 -0.23047039 -0.34875995 0.30089697 -0.18857214 -0.24328485 0.52552986 0.26533747 0.011100284 -0.5352661 -0.38449284 -0.38377059 -0.06843596 0.037823107 0.3391847 -0.1406539 -0.22526932 0.27442873 0.39005002 -0.46595117; 0.34666443 0.5118423 -0.4654628 0.13951735 0.46074197 -0.15788776 -0.025184838 -0.05037477 0.14767848 0.3725323 -0.06156609 -0.40361422 0.38895974 -0.27291286 -0.263583 0.04402309 0.36226916 -0.25727162 0.22085372 -0.03272393 -0.4284838 0.4421763 0.058823094 -0.05187229 0.28552735 0.27618372 0.49070022 -0.1365935 0.51447827 0.20009293 -0.10998602 -0.18014763 0.31233656 -0.21348129 0.40126035 -0.26607394 0.118861854 -0.21083671 0.05013624 -0.50225663 -0.30087858 -0.38753462 0.4103344 -0.121680625 0.3240014 -0.2548273 0.21176146 0.09337855 -0.11287512 -0.094466284 -0.070972346 0.043249425 -0.3924227 -0.0008175019 0.085751615 0.4068052 -0.2401978 -0.3285687 -0.0049968627 0.2781082 -0.10572981 -0.18754183 -0.53608084 0.18334876 -0.043534193 -0.078654155 0.4103525 0.22971945 -0.2012163 -0.29900908 -0.23824616 -0.19248484 0.039576925 -0.13697775 -0.14046265 -0.17788146 -0.2332769 0.37736505 -0.033755988 -0.29604167; -0.14615747 -0.16568421 0.2506423 -0.15954171 -0.39128643 0.25834164 -0.11534236 -0.3261442 -0.30342734 -0.30712143 0.33070198 0.35513717 -0.5355369 0.37701046 0.09874405 0.10014295 -0.33196998 0.2774852 -0.33698165 0.32680443 0.45434663 -0.41360536 -0.15515137 -0.09522423 -0.242046 -0.29972082 -0.19150257 0.042090848 -0.46131647 -0.40485007 0.20911792 -0.15855989 -0.26243597 0.36132228 -0.16443466 -0.03145434 -0.14656416 0.18086922 -0.09918893 0.44508547 0.20394176 0.13172989 -0.606865 0.2728159 -0.2831293 0.1569584 -0.1353288 -0.029955927 0.07547782 0.24133657 -0.1750421 -0.11428047 0.48696524 0.33571368 -0.38249 -0.44916686 -0.06748635 0.25125754 -0.15846822 -0.105987325 0.10952585 0.33485126 0.31013307 -0.12853357 0.057374906 0.22924356 -0.23117685 -0.20845705 0.24669565 0.6164473 0.19919243 0.4985359 -0.21898235 -0.20971799 -0.12145729 0.075262666 0.36697796 -0.4533543 -0.19674248 0.56790245; -0.09484291 -0.17671233 0.15090396 -0.22558185 -0.38892117 0.32716265 -0.14395636 0.12108468 -0.09667118 -0.39340568 0.283675 0.21286094 -0.3648104 0.08259036 0.33694732 -0.06304712 -0.12034172 0.29986984 -0.51077014 0.13168086 0.36949217 -0.22472484 -0.25723648 -0.3619648 -0.4188696 -0.15804036 -0.19066614 -0.12474389 -0.4466437 -0.4701565 -0.016653936 -0.22715355 -0.4640349 0.5401137 -0.19830827 -0.3483497 -0.46026948 -0.33417812 -0.44129157 0.5586857 0.5270535 0.25494796 -0.5066487 0.37038937 -0.2948285 0.15355377 -0.01255173 -0.08694043 -0.20290999 0.0069889594 0.054004125 -0.28779265 0.33979622 0.12000524 -0.43093064 -0.34232 0.1599184 0.43621302 0.3047214 -0.2147461 0.13096404 0.22771713 0.3405484 -0.23576021 0.03299925 0.25572857 -0.50953406 -0.42031506 -0.05854498 0.55877984 0.17145798 0.25879276 -0.23524857 -0.20356637 -0.21739784 -0.03981758 0.17259315 -0.15707375 -0.092013665 0.4923066; 0.24617228 0.23905352 -0.50861675 0.38413635 0.16711836 -0.07180995 -0.28557557 0.11355562 0.008993736 0.28535223 -0.27587116 -0.31190944 0.16025971 0.21238586 -0.14347017 -0.089800656 0.38473627 -0.3565764 0.3862984 0.30875424 -0.5305626 0.50568295 0.07369602 0.035442166 0.639809 0.41117847 0.04727782 -0.45398724 0.8171654 0.28961003 -0.26657182 -0.25126573 0.46710256 -0.607648 0.11865964 -0.19656093 -0.03585303 0.06755371 0.03297568 -0.3609265 -0.24537337 -0.12952958 0.47992435 0.1770243 0.059261244 -0.3029671 0.094808094 -0.30190837 -0.13670637 -0.18469332 -0.41185856 -0.31893432 -0.09453287 0.103429906 0.39916533 0.82608885 -0.15307675 -0.5209342 0.12470383 -0.25556603 0.23552404 -0.47515494 -0.45567733 -0.055308 0.5440203 -0.08255487 0.09347643 0.050282825 -0.031946257 -0.46436134 -0.5149287 -0.65486217 0.32144323 -0.19046828 0.0018655047 -0.014927054 -0.5084893 0.3196412 -0.14556694 -0.6316979; -0.05594321 -0.061860375 -0.08922645 0.10431226 0.36437404 -0.45611173 0.4049961 -0.28849447 0.52878916 0.16216011 -0.51460975 0.19122256 -0.025291774 -0.4711022 -0.24795194 0.41558883 0.21695793 -0.20391788 -0.048669115 -0.39890283 -0.26568294 -0.45416042 0.5763345 0.26960936 0.029082809 0.21960904 -0.079572186 0.46485922 0.25601047 -0.08594806 0.25762394 0.30917403 0.020017717 -0.16783361 0.23389713 0.30253556 0.467779 0.5283158 0.5581478 0.0950424 -0.35354963 -0.33189574 0.16637002 -0.057658087 0.23614192 -0.08407038 -0.18401365 0.257773 0.47533306 0.38933903 0.34903696 0.278097 -0.21437739 -0.44540846 -0.4655173 0.26437503 0.15095013 0.37630826 -0.38136294 0.26549208 -0.49752092 -0.098547846 0.17667843 0.3355058 -0.014252479 -0.51862174 0.2597962 0.29924172 0.25088745 0.033610433 -0.21496186 -0.009412507 -0.2500431 0.23172884 0.38918307 0.008949802 -0.20035253 0.36559105 0.40577188 -0.29201934; 0.13573308 0.22914638 -0.32694885 0.25109655 0.23658198 0.11574237 -0.18740945 0.15705317 -0.22415812 0.33843353 0.09418782 -0.33266136 0.555813 0.05232854 0.18906073 -0.24083741 0.39951965 -0.38250488 0.4558734 -0.13336389 0.024384163 0.4727118 -0.09691977 -0.37464508 0.26309717 0.17751174 0.26718518 -0.15246744 0.08215297 0.32720265 -0.15473494 -0.061302107 0.42624104 -0.29774144 0.16961344 -0.36693326 -0.0076825274 -0.27883667 -0.027733266 -0.5231564 -0.34919608 -0.33711725 0.1041149 -0.10585965 -0.018958965 -0.43447697 -0.07244506 0.07050883 -0.106032826 -0.27021047 -0.027477097 0.23807003 -0.13266516 0.19471656 0.14300932 0.1400191 -0.1197355 -0.29939806 0.24427705 0.17348778 0.17888111 -0.36964065 -0.23073286 -0.3239726 -0.19200186 -0.014416025 0.11256008 -0.09996674 -0.34411103 -0.19232462 -0.25783467 -0.24996711 0.21863978 0.09225164 -0.12627481 -0.35416812 -0.24045953 0.18380442 -0.29976693 -0.27376327; 0.09481758 0.20987754 -0.44315442 0.3019675 0.42210656 -0.37834287 0.07275683 -0.069451645 0.32316178 0.49500608 -0.23989245 -0.4798493 0.039390754 -0.10979956 -0.4391067 0.3267318 0.1267744 -0.5043566 0.42734015 -0.14043951 -0.26493412 0.22066376 0.3742726 0.14316747 0.19725531 0.35441417 0.06986709 -0.2705925 0.32014042 0.009535058 0.18338437 0.21530904 0.50353456 -0.43337765 0.338734 0.12893581 0.41337284 0.22875473 0.46045646 -0.5727156 -0.33030638 -0.2852127 0.47444153 -0.16506717 0.28858492 -0.2010029 0.22698906 0.03566994 0.066789985 -0.13611397 -0.08112206 0.31163144 -0.26174867 -0.16711836 0.2889659 0.38325548 -0.10173115 -0.16431981 -0.29235435 0.4164182 -0.48135948 -0.21289578 -0.33711806 0.41513297 -0.06613027 -0.42189837 0.5940276 0.5228906 -0.04317006 -0.5405084 -0.47934073 -0.5816676 0.11651067 0.1505771 0.5078684 0.04610332 -0.58338726 0.43038556 0.24594228 -0.58003503; -0.22399223 -0.14800362 0.36312047 0.11251671 -0.0034746372 -0.35164472 0.44770384 -0.10869298 0.1939943 0.21442051 -0.27628312 0.31489983 -0.41662717 -0.2232468 -0.042460095 0.31694323 -0.24825199 0.2786592 -0.34216252 -0.3370704 -0.33091745 -0.48409557 0.031049235 0.20341277 -0.3840523 0.23402515 -0.49108613 0.1489486 0.19180183 -0.15220541 0.22346628 0.2451873 -0.2508008 -0.2657118 0.052132357 0.31383604 0.3880803 0.24928473 0.3720771 0.36206025 0.08143054 0.03458536 -0.3233425 0.2791852 0.17398092 0.076303184 -0.35220736 0.06337486 0.30687436 0.11532157 0.123662144 0.11871834 0.020407973 -0.17959556 -0.2904291 -0.13513216 0.13151857 0.32295123 -0.4929386 -0.05570286 -0.07444101 0.049955316 0.36152625 0.22366446 0.2433665 -0.46227077 0.0033494579 0.35434735 0.10487005 0.0468008 -0.1305312 0.10576078 -0.30368486 0.30489957 0.52081805 0.017547943 -0.1573626 0.019632867 0.13761656 0.20773491; -0.21294552 -0.12772262 0.24108398 -0.4100015 -0.4526354 0.2741961 0.1783916 -0.32862037 0.03393182 -0.44533053 0.3289828 0.30100608 -0.26910567 0.20543687 0.08476707 0.098067656 -0.23092934 0.49614272 -0.23102029 0.11715186 0.27447912 -0.27897793 0.12138676 0.10323592 -0.48223644 -0.5867307 -0.03924468 0.30367067 -0.43401337 -0.43082616 0.12788501 -0.049592216 -0.5650925 0.3212011 -0.42797717 -0.1081602 -0.33686975 -0.090208754 -0.039250664 0.5174508 0.4368862 0.13287924 -0.5180661 0.20818074 -0.13065057 0.14883587 -0.2935812 -0.027933711 -0.070069596 0.24631692 0.18945174 -0.032464262 0.4306545 0.17951748 -0.5767069 -0.5418705 -0.16972691 0.13268587 -0.250314 0.030721378 -0.09808246 0.31989902 0.34445983 -0.21563295 -0.1383487 -0.0035921852 -0.19806834 -0.24349985 0.06783549 0.71114624 0.40762302 0.4629744 -0.01329189 0.040337212 0.019359846 -0.12815292 0.6251791 -0.4244922 0.052565854 0.7445441; 0.055448886 -0.15363991 0.46382546 -0.18808122 -0.30962467 0.18098426 -0.21507905 -0.056514677 -0.38543802 -0.38492805 0.48713434 0.45535582 -0.36854863 0.24793734 0.4804948 -0.28147742 -0.31651908 0.32421926 -0.34960213 0.3470477 0.44998452 -0.40659037 -0.2021198 -0.14471644 -0.3353779 -0.28920427 -0.06089669 0.18795966 -0.29918063 -0.1733589 -0.04216898 -0.11689658 -0.55610657 0.5005969 -0.4663674 -0.30022797 -0.52527964 -0.07487473 -0.26400894 0.25592577 0.24995363 0.21492924 -0.56472075 -0.02701583 -0.5070245 0.066649966 -0.13965468 -0.27605444 -0.32239598 0.088156745 -0.008495285 -0.39674878 0.33753267 0.18770081 -0.1974986 -0.45146245 -0.04935125 0.2900033 0.008365702 -0.3503097 0.2064031 0.4360765 0.4239409 -0.40861985 0.050349005 0.17705911 -0.39273277 -0.34417787 0.1248773 0.5398656 0.29540884 0.34993008 0.05491037 -0.12357083 -0.11849853 0.24083063 0.26209658 -0.36581331 -0.36983 0.52870864; 0.23261501 0.016337268 -0.24221866 0.14876579 0.3031736 -0.38080087 -0.020173352 0.34440485 0.08912238 0.5259755 -0.50446415 -0.5999781 0.15166129 -0.24797074 -0.45961216 0.19448642 0.44100425 -0.31499714 0.23815548 -0.28517652 -0.42790234 0.2983479 0.35118046 0.30344415 0.36895934 0.4662537 0.31176284 0.018506624 0.61101663 0.17627364 0.17758128 0.22801341 0.59115505 -0.4164563 0.28957286 0.21206336 0.28420988 0.25878266 0.3115303 -0.3835741 -0.356318 -0.17922635 0.34073803 -0.055831693 0.2776095 -0.05967756 -0.0028379182 0.23526552 0.2869993 -0.19679874 0.08940841 0.3413947 -0.1752103 -0.38765812 0.27386072 0.4254229 0.06763112 -0.20687082 0.011146778 0.17992148 -0.21883395 -0.5100632 -0.4150598 0.13034523 0.025256738 -0.28935003 0.5951191 0.22608109 0.0650416 -0.68005574 -0.22147143 -0.43641427 0.012055565 0.1906026 0.37019217 -0.23481429 -0.35579073 0.34945467 0.37295696 -0.65104884; -0.2584788 0.05058718 -0.32814378 0.22378023 0.29203162 -0.28927466 0.44757873 -0.08276071 0.4904633 0.2353489 -0.48921758 0.07669787 -0.53887105 -0.5548447 -0.35481226 0.25607184 0.20711088 -0.21844317 -0.4238789 -0.30530038 -0.20107412 -0.20723774 0.4906613 0.46048382 0.057787646 0.47326607 -0.05541563 0.08756041 0.37732238 -0.3099191 0.29312786 0.24944222 0.17107375 -0.32847086 0.38937908 0.46583194 0.17013527 0.4410191 0.4111509 0.058015145 -0.29443896 -0.38696828 0.2172859 0.048420105 0.6454928 -0.122153506 -0.05275397 0.2535657 0.4779599 0.32441407 0.30885988 0.41828203 -0.17303424 -0.27397206 -0.1819986 0.170835 0.11875551 0.23327781 -0.5119826 0.560129 -0.2950257 -0.093956135 0.3059437 0.4143765 0.013710587 -0.49548998 0.29857543 0.45976058 0.17698532 -0.13265721 -0.4923583 -0.34444422 -0.291853 0.1352338 0.4879861 -0.004882673 -0.39975345 0.41486445 0.30269596 -0.14493047; 0.17707717 0.36612886 -0.19911595 0.33297807 0.25272906 -0.23952559 0.13671641 0.008333073 0.054227646 0.2911127 -0.31594038 -0.26794037 0.31422022 -0.38890272 -0.26189584 -0.019285494 0.15756598 -0.27277863 0.51770794 -0.16724522 -0.4927858 0.5004458 0.15120457 0.046556104 0.308569 0.49871606 0.39485252 0.007910139 0.2919434 0.17537037 -0.26169926 -0.03555862 0.24272892 -0.20612887 0.43947542 0.17904927 0.35314336 -0.026704665 0.13271278 -0.21423395 -0.27423453 -0.35879332 0.3777411 -0.16682255 0.436818 -0.122624226 0.11145085 -0.022123812 0.20319031 -0.25012675 0.02213609 0.26480427 -0.3368956 -0.04919909 0.34188014 0.35346502 -0.053199135 -0.26358646 0.10752556 0.121609256 -0.30245304 -0.1728054 -0.36852372 0.0011251005 -0.05753908 -0.35328776 0.38275367 0.13453327 -0.3022103 -0.6055442 -0.21956834 -0.20330532 0.23285289 -0.06657063 0.3929999 -0.07934301 -0.43420428 0.41283053 0.15326244 -0.28187552; 0.15534954 0.008982352 -0.35265845 0.3484616 0.38328904 -0.4600051 0.0054513975 -0.038345672 0.38759923 0.48559326 -0.21699592 -0.38754755 0.2413633 -0.38349822 -0.18910627 0.40895984 0.30704105 -0.5426572 0.42372242 -0.31507468 -0.50776184 0.29888996 0.4341883 0.09182742 0.24683706 0.42678055 0.1805113 -0.072042584 0.46876192 0.17066395 0.09722805 0.10577982 0.29956067 -0.5765573 0.33788174 0.073284365 0.3260729 0.3442128 0.20813607 -0.27724588 -0.28558093 -0.1568984 0.61826384 -0.16991997 0.3104139 -0.17061411 0.16723402 0.031066766 0.44866103 -0.24076961 0.27442297 0.29153755 -0.17400602 -0.39916486 0.3596541 0.5518344 -0.10195607 -0.2392153 -0.042207733 0.23556095 -0.50777066 -0.29107156 -0.33319885 0.3078093 0.18734504 -0.5093308 0.3072735 0.24815744 -0.0038735026 -0.6320094 -0.235273 -0.47871503 -0.12969647 0.226209 0.43668664 -0.03916764 -0.37104815 0.49187627 0.2358637 -0.624331; -0.36299607 -0.33714122 0.47228563 -0.17419037 -0.30766827 -0.1751959 0.061004877 0.044345886 0.021323605 -0.26001567 0.095518306 0.3807977 -0.5434639 0.0004522722 -0.07822718 -0.0077800374 -0.35073307 0.18336202 -0.3174852 0.2001799 0.21881656 -0.38764268 -0.0732482 0.05486391 -0.58485156 -0.48452157 -0.32728645 0.10445324 -0.2131165 -0.52288586 0.35034433 0.18722023 -0.25451687 0.43691358 -0.19355321 0.274867 -0.12967324 0.30426717 -0.013153896 0.28714582 0.15511775 0.1715171 -0.40410966 0.34820962 0.026956735 0.48014572 -0.00044254074 0.107115366 0.21907826 0.31617892 0.016345188 0.058780745 0.1498915 -0.006321171 -0.36856034 -0.30344573 0.15610908 0.26418915 -0.15026887 -0.040999923 -0.12850143 0.22304124 0.26220587 0.060166508 0.081311874 -0.13905802 -0.02078254 0.16087689 0.15606001 0.41805273 0.24894653 0.16686548 -0.22306015 0.03604279 0.21981181 0.2379075 0.2923314 -0.41890725 -0.07812724 0.46360365; 0.11216877 0.18576595 -0.1535126 0.41359684 0.20960504 -0.40519977 0.3630913 -0.09212653 0.4090296 0.16966148 -0.47840837 -0.5153188 0.21480411 -0.21837518 -0.35452846 0.12192341 0.46157137 -0.30274868 0.43621248 -0.12545927 -0.24727826 0.4167257 0.26757142 0.01442815 0.15975645 0.3119134 0.4739544 -0.2138214 0.39656773 0.37946603 0.056293536 0.11476387 0.21455628 -0.25811207 0.43040806 -0.00056015985 0.48151135 0.19974238 0.08399789 -0.2727304 -0.5298627 -0.26878512 0.298475 -0.11674982 0.11815619 -0.15852638 0.028577682 0.038934812 0.22703794 0.00757186 -4.6466776f-5 0.15401311 -0.11823727 -0.110369086 0.11749748 0.3192698 -0.14511056 -0.5058381 -0.2444361 0.13667685 -0.1456123 -0.50167596 -0.45141515 0.11675365 -0.02784701 -0.39798248 0.3967324 0.28930658 -0.03645414 -0.6315334 -0.14928922 -0.3994362 0.16571459 0.27932882 0.38445568 0.008055813 -0.5547491 0.166068 0.23862785 -0.595044; -0.034389272 -0.19112813 0.58013743 -0.4005994 -0.41518903 0.20080386 0.14757341 -0.26257044 -0.12073168 -0.41953292 0.1896359 0.40435603 -0.1453907 0.10623356 0.16767442 -0.08216252 -0.3652437 0.3631388 -0.17050946 -0.047291394 0.51622146 -0.4901719 0.005637471 -0.08090224 -0.18878093 -0.3225148 -0.1189857 0.15847479 -0.29048404 -0.38073173 0.26591522 -0.0846346 -0.3234921 0.3390945 -0.21415077 0.13090882 -0.350909 -0.019297535 -0.09559047 0.4467581 0.39783812 0.36041626 -0.6514051 0.0306995 -0.4195488 0.39941987 -0.25299683 0.14060098 0.063860334 0.21455035 0.28800645 0.011383742 0.22802967 0.02700005 -0.30641234 -0.65279436 0.09079702 0.38048327 -0.08863955 -0.35245794 0.124299675 0.3563998 0.4129897 -0.225315 -0.2815248 0.15537685 -0.19130659 -0.1307191 0.06566471 0.39872205 0.29002175 0.5254033 -0.28197247 0.034727953 -0.0061792824 0.08256086 0.3263495 -0.33793035 -0.20710559 0.44627178; 0.4772536 0.25380182 -0.21059683 0.27225083 0.4501371 -0.09637602 -0.16360722 0.1378211 0.1340192 0.29673946 -0.3029621 -0.25107318 0.43472484 -0.07023463 -0.1272649 -0.17310295 0.16451631 -0.24266222 0.48812303 -0.188205 -0.34760368 0.28509894 -0.0007427973 0.0643427 0.3020035 0.15173526 0.27077693 -0.0011315627 0.37649828 0.33151636 -0.15309921 -0.32160118 0.3396141 -0.3541336 0.39301583 -0.008696469 0.13156307 -0.12254986 0.06687469 -0.46157214 -0.30481014 -0.20838347 0.51268435 -0.17386758 0.0825052 -0.21993728 0.13843203 -0.15457839 -0.0820728 -0.028688094 -0.05028254 0.10585276 -0.3546438 -0.039377343 0.49087822 0.30151305 -0.2095221 -0.25021023 0.2403216 0.26477444 -0.06426929 -0.30823314 -0.54396856 0.17676634 -0.042603534 0.095236704 0.17900608 0.18803251 -0.22771682 -0.59720576 -0.1769427 -0.5683308 0.32553214 0.15454459 -0.22540618 -0.120549746 -0.359979 0.24856822 -0.03408325 -0.3322411; 0.08900275 0.2167277 0.29252666 -0.27463916 -0.4250565 0.32649687 -0.2613718 0.16827534 -0.542423 -0.23458572 0.25564092 -0.09019995 0.24534842 0.45358115 0.49572662 -0.42670748 -0.023410305 0.44017348 0.29466286 0.42961356 0.4138948 0.34004515 -0.34486595 -0.540313 -0.078411974 -0.15396379 0.092893705 -0.2816709 -0.15628901 0.10702709 -0.3971327 -0.29634032 0.12620312 0.3846139 -0.20707336 -0.29134956 -0.483924 -0.34413883 -0.31522712 0.04575098 0.17954436 0.17183784 -0.017195892 0.27070394 -0.42842704 -0.07757801 0.37858373 -0.37212265 -0.48173496 -0.30231607 -0.46155202 -0.58283603 0.16628708 0.26658 0.44743732 -0.28780824 -0.007543292 -0.558544 0.34452912 -0.21943095 0.2859839 0.24733683 -0.22214377 -0.44518805 0.12274669 0.54320025 -0.39400843 -0.2982499 -0.024063252 -0.028318789 0.16160241 0.35411674 0.416689 -0.33743382 -0.5489521 0.2610463 0.36660504 -0.2208187 -0.17924027 0.063615695; 0.052154955 -0.20019108 0.36115816 -0.3485694 -0.27639207 0.45224822 -0.039246224 -0.29241836 -0.20411424 -0.3196843 0.20877902 0.55748886 -0.13631442 0.08185759 0.41185364 -0.38516772 -0.475675 0.17993808 -0.5298394 0.21200004 0.29470623 -0.36365542 -0.19151105 -0.07484441 -0.4217314 -0.21947488 -0.21093257 -0.029615805 -0.49555182 -0.22365864 -0.10816386 0.025395297 -0.48675168 0.47099954 -0.21393207 -0.3011049 -0.47500148 -0.017493967 -0.33299935 0.26853415 0.53677946 0.4660456 -0.58741474 0.30871677 -0.17084417 0.33756578 0.0020179222 0.012421134 -0.3708109 0.29158914 -0.056299236 -0.1103478 0.45583987 0.16483778 -0.4066452 -0.25065336 0.13928801 0.2095679 0.07979237 -0.32637838 0.44083825 0.30681252 0.49407122 -0.062694766 0.22048168 0.20804414 -0.51999176 -0.47858474 -0.016788358 0.4426708 0.28348285 0.3393734 -0.102134585 -0.06797644 -0.23206393 0.1913656 0.37088245 -0.2739182 -0.21740513 0.32686642; 0.18734455 0.26621377 0.21184024 -0.4557708 -0.49176624 0.59425807 -0.27132156 0.24337353 -0.36568338 -0.5225301 0.42701948 0.094547644 0.14256108 0.40813807 0.52981406 -0.35200822 -0.077967584 0.45135015 0.0977545 0.5362584 0.27778685 0.2767971 -0.62135375 -0.5745296 -0.054544713 -0.24506648 0.14019804 -0.33758467 -0.16209736 0.29902127 -0.21809836 -0.56517285 -0.15034334 0.27447623 -0.35625076 -0.47150004 -0.40183088 -0.3797688 -0.4511729 0.1119935 0.5249547 0.3508695 -0.22586083 0.11490321 -0.65118426 0.16975893 0.32537478 -0.25095665 -0.47825882 -0.5482637 -0.28488034 -0.44473305 0.07356877 0.25752082 0.17885372 -0.38499844 0.030921066 -0.1664089 0.33936518 -0.27548224 0.43690413 0.17531255 -0.12708783 -0.47420272 0.13228558 0.47077936 -0.29243332 -0.28003755 0.035615202 0.22727785 0.5890442 0.4963861 0.07662266 -0.26076102 -0.5424385 -0.11793893 0.5189167 -0.28600228 -0.28044325 0.39799684; -0.24875444 -0.4140525 0.24076909 -0.15263109 -0.10771722 -0.120831124 0.1481797 -0.10622769 -0.08538569 -0.46661165 0.27105185 0.563645 -0.41540942 0.25672275 0.14487258 0.17410429 -0.33504954 0.4246489 -0.5040535 -0.017590765 0.44734764 -0.24826278 -0.09447348 -0.007008599 -0.44560727 -0.31871057 -0.203698 0.05867407 -0.41144156 -0.33581066 0.27212992 0.14211228 -0.3320103 0.2742731 -0.3429919 0.04585765 0.08186886 0.23736835 0.05017972 0.4544601 0.2431827 0.42120963 -0.42330098 0.20090534 -0.33569512 0.39874566 -0.32705453 0.10840394 0.2844771 0.31764075 0.032169133 -0.1960673 0.18607794 -0.04215967 -0.46311632 -0.51766443 0.03908 0.3031552 -0.012921133 0.03707622 -0.13621555 0.31892648 0.21686742 -0.14551714 0.038158208 -0.12931357 -0.11797658 0.0679021 0.30082732 0.42981935 0.2598168 0.42948216 -0.19261557 -0.052437294 0.10341921 0.137611 0.43502453 -0.23133588 0.19532944 0.5676589; 0.45757538 0.07048048 0.013800093 -0.24620265 -0.4025129 0.4240675 -0.4536435 0.088256516 -0.32457483 -0.24831353 0.35366225 -0.054431282 0.49608102 0.41583768 0.28258613 -0.31723267 -0.10203844 0.060819972 0.36653626 0.46862966 0.43657604 0.6131401 -0.48689914 -0.24696933 0.18212113 -0.45244688 0.12174373 -0.11141199 -0.35278815 0.17594725 -0.20894507 -0.5501988 -0.09368485 0.20911798 -0.34289867 -0.48810574 -0.13974324 -0.3389566 -0.3883957 0.0859488 0.11019868 0.35063672 0.16738977 0.11084099 -0.66046256 -0.26972696 0.10050633 -0.1844924 -0.38409048 -0.27899247 -0.51354975 -0.49328604 -0.10213919 0.30946392 0.11882511 -0.08101968 -0.23335804 -0.3218301 0.20262675 -0.39644203 0.28775072 -0.029513905 -0.07922301 -0.36308083 0.15352707 0.29253033 -0.47515702 -0.26275453 -0.12361684 -0.16657807 0.5697157 0.064378984 0.23635481 -0.24826895 -0.27999318 0.062262207 0.4343391 -0.3425996 -0.21836041 -0.0049537667; 0.1795288 0.2130759 -0.25621098 0.15440457 0.17334345 0.15985893 -0.071653455 0.048971605 -0.29860586 0.11654428 0.056239754 -0.47940686 0.63214827 0.10545208 -0.09028195 -0.061429564 0.45675054 -0.47810432 0.30889696 0.18958527 -0.1065052 0.36022943 -0.32378978 -0.36965844 0.5427094 0.29350236 0.25616163 -0.28179514 0.104172744 0.29582378 -0.4538114 -0.16931535 0.5082757 -0.19324222 0.22391628 -0.21022584 -0.020826392 -0.07895073 0.03741051 -0.49309894 -0.35511988 -0.033871356 0.18511766 -0.18634328 -0.13476384 -0.22958499 0.24445894 -0.08529059 -0.15620156 -0.03316832 -0.23180649 -0.031385396 -0.5083695 0.25642687 0.10082762 0.35473365 -0.45369264 -0.43011892 0.22973134 -0.017278925 -0.034312297 -0.41290253 -0.42723423 -0.07988781 -0.36098576 0.31559476 -0.07040315 -0.12580876 -0.36188158 -0.16697519 -0.38119468 -0.43186346 0.30556196 -0.013460484 -0.08732503 -0.40099868 0.027044898 0.20700222 -0.25476676 -0.15801683; -0.11176339 -0.16334179 0.2721776 -0.48357722 -0.5738147 0.25220302 -0.065097384 -0.18372516 -0.25050822 -0.56939465 0.25369754 0.40861872 -0.23674563 0.34580317 0.30595627 -0.07457093 -0.25440177 0.25641614 -0.19394615 0.34312835 0.48434693 -0.18031286 -0.37676954 -0.38190642 -0.31900087 -0.34458575 -0.06298216 -0.0868533 -0.56171244 -0.061032124 0.15193681 -0.10222223 -0.21339571 0.26886284 -0.22330551 -0.34064692 -0.07499867 -0.23451589 -0.4169375 0.5107764 0.5016698 0.20953932 -0.38677797 -0.011787137 -0.32031178 0.17012666 -0.042412315 -0.05312107 0.010244179 0.21228057 -0.246729 -0.39530692 0.09958253 0.24353412 -0.25849766 -0.52851295 -0.16637737 0.33157697 0.16086699 -0.35261756 0.23251016 0.30823502 0.2646406 -0.28711602 0.05232342 0.3371258 -0.41223508 -0.18675505 0.18727313 0.22290716 0.33290178 0.39137924 0.028648304 -0.12858565 -0.1018694 0.20387468 0.41160658 -0.30651176 -0.3654613 0.31241068; -0.37094998 -0.27077365 0.17496714 -0.37153304 -0.30502334 0.2673479 -0.0007606139 -0.05799696 -0.3008676 -0.20681928 0.1394166 0.248705 -0.40172848 0.2569045 0.14001326 -0.15673108 -0.41553813 0.3732414 -0.39089662 0.18407418 0.4357878 -0.16649435 -0.036088526 -0.2784587 -0.4020158 -0.29455176 -0.4896239 0.06210566 -0.3378404 -0.50348026 0.04232226 -0.0076947236 -0.5299303 0.24386121 -0.17859967 -0.30970433 -0.13762125 -0.27283913 -0.23454255 0.31156573 0.22057226 0.43885344 -0.5054002 0.20518315 -0.09392684 0.15995564 -0.023259059 -0.051889654 -0.08894572 0.082043454 -0.00012680303 -0.26844138 0.27663663 0.37028024 -0.21967168 -0.55248255 -0.12015968 0.37952134 0.13122614 -0.19387385 0.22587316 0.46749884 0.41871274 -0.34323108 0.16064304 0.12380334 -0.48026925 -0.34727803 0.27432162 0.49332005 0.13079405 0.47612906 -0.06796244 -0.35549414 -0.41423896 -0.01041706 0.3736023 -0.1322944 -0.12913942 0.4253627; 0.20998582 0.1217163 -0.5016537 0.29065442 0.3737979 -0.27024144 -0.00991644 0.07430251 0.038222704 0.4736789 -0.1647442 -0.61011004 0.50213283 -0.021762272 -0.23239596 -0.06449974 0.16764069 -0.5782672 0.46973118 -0.17799658 -0.46735415 0.1837395 -0.16147727 -0.046119038 0.36018094 0.24472861 0.24300466 -0.082893334 0.37171724 0.26145205 0.028667253 -0.026184255 0.40659598 -0.506237 0.2300185 -0.23999463 0.27743334 0.004626414 0.26161546 -0.3177885 -0.54575926 -0.48469114 0.35026708 -0.001625607 0.26681343 -0.20357892 0.4127005 0.059459243 -0.029114975 -0.1274577 -0.29911092 -0.11044844 -0.15598325 -0.20568714 0.550488 0.63939935 0.1513893 -0.379077 0.033972625 0.27043426 0.059937518 -0.5577888 -0.36654672 -0.16658925 0.37565932 0.16759056 0.5026481 0.05750083 0.022916911 -0.42001864 -0.33310896 -0.33655298 0.20714806 -0.075808294 -0.052849516 0.048542887 -0.6130581 0.46364614 -0.120102316 -0.50551903; 0.19623384 -0.06816692 0.16492695 -0.3796024 -0.30659375 0.25900066 -0.43087432 0.35503608 -0.39991352 -0.19755328 0.46504644 -0.058254585 -0.03297105 0.22779806 0.51536024 -0.5777517 -0.20852031 0.22384 0.17477588 0.22862686 0.38743278 0.29104534 -0.27763575 -0.29231954 -0.097954705 -0.23007478 0.10407544 -0.12181514 -0.39200303 0.0570592 -0.23012869 -0.29974857 -0.28896257 0.14855243 -0.32916546 -0.47011194 -0.22333787 -0.47588313 -0.46517822 -0.013555541 0.48214406 0.18008587 -0.08621291 -0.018331952 -0.49499485 -0.09106597 0.28192043 -0.3869182 -0.38503405 -0.46962044 -0.27181214 -0.4591382 0.01201108 0.32461753 0.39354175 -0.2830622 0.045840874 -0.24048199 0.42813507 -0.2629005 0.38702494 0.2624502 -0.03284551 -0.28958535 0.19087045 0.5126991 -0.48862794 -0.5098802 -0.021496143 0.054153245 0.43141595 0.43616313 0.3608174 -0.27223915 -0.55264443 0.033781998 0.40219182 -0.23821835 -0.23314032 0.29974943; -0.1174879 0.19386746 -0.22789828 0.34183583 0.22601233 -0.17197055 0.55926883 -0.32687655 0.18903142 0.31916112 -0.035253763 0.16525155 0.22072037 -0.49170896 -0.32879192 0.35958037 0.39177847 -0.45057142 -0.2060599 -0.46491048 -0.03523321 -0.07771489 0.5280839 0.0848661 0.1342283 0.28044152 0.04330391 0.32748076 0.13989697 0.0896315 -0.06967295 0.2985188 0.29631054 -0.08899678 0.4080151 0.44000927 0.2643482 0.30555472 0.60107 0.13692373 -0.2370196 -0.34800464 0.17214958 -0.23924735 0.45443928 -0.440202 -0.5297136 0.49159497 0.35468176 0.48603094 0.6103924 0.3815631 -0.1852807 -0.41765022 -0.54193634 -0.08360332 -0.34489244 0.22685395 -0.29719356 0.56118363 -0.39387706 -0.326834 -0.09647907 0.13606665 -0.35257462 -0.18322693 -0.010477124 0.2883854 -0.001942246 0.13134812 -0.18445559 -0.15445517 -0.24056207 0.6221686 0.19358343 -0.20889263 -0.042096615 0.25445923 0.23144594 0.15586148; 0.25796565 0.004501252 -0.5419639 0.32212687 0.49667588 -0.14777382 0.09731358 0.26603526 0.22300182 0.5268033 -0.2666566 -0.4180058 0.38156137 -0.08677756 -0.33844292 0.038591377 0.23343347 -0.30045938 0.43509054 -0.18516713 -0.5051792 0.3168224 0.1431061 0.30294698 0.46324238 0.565177 -0.005133335 -0.033665366 0.50078654 0.17534102 0.0068755937 -0.09823057 0.5449816 -0.5002402 0.14798155 0.18200895 0.51313376 0.20389211 0.27067447 -0.5195517 -0.41846183 -0.31713825 0.5130303 -0.1523358 0.26005828 -0.03143768 0.16590965 0.16006292 0.19024351 -0.20617065 0.04587804 0.18825795 -0.4398467 -0.3356671 0.3331876 0.37784144 0.12870531 -0.44732788 -0.2212613 0.29156667 -0.13473336 -0.4193956 -0.46390483 0.119228706 -0.044601157 -0.36051777 0.49663186 0.47536448 0.13484167 -0.3709423 -0.3698848 -0.35269707 -0.09707577 0.07946141 0.18278788 0.014521136 -0.29001462 0.2954828 0.29466343 -0.512288; -0.49179217 -0.18828587 0.35010442 -0.40488312 -0.37242943 -0.24305788 0.13253851 -0.24438533 0.33337098 -0.13651232 -0.14140683 0.3138517 -0.519639 -0.24612471 -0.16414389 0.028970625 -0.46775848 0.16975561 -0.31512055 0.033318087 0.21952006 -0.25098774 0.021218138 0.35511452 -0.5787109 -0.33341283 -0.3095253 0.35704195 -0.026469141 -0.32711172 0.36587837 0.20736314 -0.44878003 0.070127934 -0.25566483 0.0057543195 -0.040829044 0.33824798 0.25131726 0.27211544 0.2585004 0.14784993 -0.21598668 0.07248167 -0.05946866 0.37883347 -0.07918622 0.042106874 0.3052406 0.14148341 0.13357 0.07255686 0.4812139 0.11439724 -0.25163868 -0.18054132 0.36027017 0.33751458 -0.2123724 0.08441861 -0.110988684 0.56211936 0.24282438 -0.0003839781 0.077879876 -0.1620174 0.1164611 0.020387348 0.32243907 0.39338222 0.35916618 0.39870217 -0.3854962 0.19371253 -0.009779202 0.27259097 0.24960308 -0.10601768 0.09698446 0.19073582; -0.44331852 -0.3601543 0.47773546 -0.4335764 -0.25583962 0.01935684 0.026476098 -0.24033907 -0.027808147 -0.30562395 0.12304848 0.59842753 -0.40810207 0.055597924 0.0196663 0.079896756 -0.2105244 0.36536923 -0.41508454 -0.05506018 0.22890149 -0.32792598 0.12418265 0.110089496 -0.30858737 -0.21736388 -0.34705073 0.15925965 -0.4353023 -0.36701205 0.22129442 0.15898697 -0.29692233 0.40268067 -0.4486097 0.07552959 -0.06059888 0.0036531254 0.07675434 0.61337644 0.3771212 0.17048451 -0.2858766 0.40092072 -0.042169105 0.28714454 -0.024521068 0.16015393 0.17164716 0.08513999 0.07354126 -0.060301773 0.37115806 0.03861355 -0.31577563 -0.23749635 0.14786841 0.44640693 -0.0189547 -0.1173134 0.0077204187 0.50982416 0.34129936 0.20834735 -0.07969955 -0.10211957 -0.04078845 -0.14819652 0.3541518 0.38570267 0.51219255 0.3609342 -0.38430107 0.23205441 0.02664989 0.15652022 0.21889241 -0.4416029 0.016638193 0.4841746; -0.32171115 -0.106417164 -0.022400714 0.14034715 0.30281648 -0.17160401 0.56808823 -0.09344928 0.15654145 0.31933886 -0.111465946 0.32165357 -0.38209093 -0.3625419 -0.16925026 0.35099033 0.099846974 -0.04763169 -0.39116567 -0.3446773 -0.09188356 -0.18351997 0.5293556 0.14342664 -0.29798383 0.03047756 -0.17309499 0.22856802 0.074096054 -0.109364204 0.12344293 0.49384326 -0.045984287 -0.33438003 0.23403656 0.24602923 0.26975137 0.2673438 0.36431715 0.20063376 -0.14293781 -0.43498883 -0.24757545 -0.20604011 0.6169961 -0.10993515 -0.28315967 0.30190605 0.45801085 0.29038754 0.51702017 0.28716904 0.004435279 -0.54295117 -0.25512743 0.13394721 -0.009177847 0.26854014 -0.2801578 0.26101443 -0.23399006 0.13657187 0.026547465 0.31031573 -0.26472518 -0.5161113 0.06519029 0.23200512 0.007603888 0.09293578 -0.33477485 0.15870962 -0.43390906 0.3017255 0.47766814 0.13303125 -0.26846173 0.011613656 0.31889874 0.2316136; 0.12280781 0.39418483 -0.38653046 0.44282645 0.25849932 -0.17045066 -0.055746276 -0.060034014 -0.046084892 0.36712092 -0.38798 -0.44758013 0.4041341 -0.05491406 -0.08258336 0.24394453 0.13651301 -0.256326 0.53814644 -0.28408852 -0.45424375 0.2715525 -0.024381861 0.20185453 0.38124686 0.21156861 0.23382327 -0.11956134 0.47696853 0.20061986 -0.099404864 -0.1618066 0.5962717 -0.3097644 0.19123834 -0.064479314 0.38491207 0.01657567 0.1193222 -0.5624905 -0.30449963 -0.34945878 0.56831056 -0.105034046 0.14490114 -0.27654976 0.098123506 0.073416434 -0.08728939 -0.2972869 0.10025853 0.08844737 -0.44962943 -0.19270763 0.08079233 0.4280577 -0.108262464 -0.28734118 -0.05254704 0.05529049 -0.0905222 -0.3016498 -0.3795867 0.30312526 -0.05855441 -0.06701467 0.24575849 0.37904784 -0.30914333 -0.27518982 -0.26426023 -0.42004004 0.08261388 -0.0004009104 0.29013303 -0.2379494 -0.3367018 0.25212076 0.17671183 -0.4573951; -0.14467923 0.14561528 -0.465259 0.2531684 0.33767644 -0.46761486 0.3282352 -0.0029771195 0.2536681 0.3832334 -0.52562785 -0.3156206 0.22671962 -0.42757887 -0.45354578 0.441778 0.19630699 -0.27013838 0.09364484 -0.45821145 -0.39481795 0.10168078 0.3817314 0.43021953 0.05476367 0.20116344 0.17482637 0.24179703 0.45998493 0.015236285 0.04917003 0.38095286 0.17876609 -0.18683392 0.19313936 0.42804655 0.46961412 0.52903384 0.24416031 -0.30720827 -0.38542125 -0.117471375 0.30049494 -0.31255654 0.36631972 -0.24761814 -0.21506406 0.024480296 0.2513346 0.12262861 0.41318497 0.1228063 -0.18528661 -0.13900648 0.1625927 0.40613732 0.079646185 0.053611316 -0.45239106 0.2660881 -0.40319455 -0.27577594 -0.3224475 0.51523775 -0.09946087 -0.21211933 0.4586285 0.43847692 -0.12844063 -0.20158693 -0.18974139 -0.4215154 -0.2784023 0.09801322 0.39723524 0.008793168 -0.18956567 0.1505114 0.24175654 -0.5183288; 0.13245168 0.0742878 -0.40621248 0.3636071 0.13615337 -0.30811822 0.025697969 0.32991463 0.08148482 0.3148542 -0.41195112 -0.5620357 0.22259492 -0.2797928 0.002633927 -0.18952009 0.21943961 -0.41210032 0.391581 -0.20323081 -0.41682684 0.20621549 -0.17373368 -0.20598397 0.4965846 0.37830263 0.37954703 -0.3444837 0.45944402 0.40077037 -0.039850783 -0.19484359 0.44403407 -0.47397175 0.31517372 -0.039069675 0.13188492 -0.18056035 -0.07376442 -0.40018404 -0.27007467 -0.3769548 0.40709472 -0.23218955 0.033696625 -0.14990272 0.10356732 0.11345909 -0.072097205 -0.24621588 0.02608691 -0.03882964 -0.4559113 -0.21554701 0.37337872 0.6257928 0.1673306 -0.30931282 -0.0031296802 0.1568731 0.046751108 -0.4012464 -0.19161057 0.16199948 0.07643164 0.0932928 0.2984553 0.30169308 -0.03885996 -0.5760087 -0.21132827 -0.26234376 0.16103458 -0.30247694 0.18011685 -0.05711017 -0.53581005 0.38683894 0.18870674 -0.61487466; -0.30332413 -0.09765809 0.5536596 -0.25969478 -0.17710637 0.13292903 0.36469933 -0.23661128 -0.036863886 -0.45074016 0.03448633 0.28807786 -0.39032704 -0.22383526 0.1288712 0.113222666 -0.27983654 0.51111895 -0.29317155 0.05106021 0.47477272 -0.24995038 0.15815657 0.20554264 -0.22067347 -0.4177385 -0.2601032 0.32117078 -0.3365985 -0.12201252 0.27457666 0.10479062 -0.42519268 0.48436332 -0.40937728 0.03377233 0.1069869 0.18101479 0.0548333 0.33405197 0.3389375 0.20321257 -0.45792305 0.09237422 0.054626774 0.16520622 -0.17750686 0.09696558 0.2353203 0.0916404 0.34906888 0.04934036 0.08761556 -0.16195497 -0.36126786 -0.38348 0.032279585 0.3161455 0.014052255 0.18193772 0.0715612 0.43694627 0.43943512 -0.023258118 -0.2816291 0.08261402 -0.36615714 0.02886203 0.05322511 0.5385491 0.37658742 0.545972 -0.20463736 0.12831178 -0.08833976 -0.17797759 0.6242599 -0.43688825 0.23707645 0.6042837; -0.24526289 -0.27193078 0.4686567 -0.39863807 -0.38863048 0.18369545 0.040057063 -0.3656755 0.07790757 -0.3905257 0.1845772 0.42164326 -0.36290082 0.006430492 0.03201249 0.1456489 -0.22703522 0.19924162 -0.40907392 0.060736228 0.31500742 -0.31717062 -0.10358375 0.06451129 -0.5455054 -0.3796206 -0.3932219 0.29546818 -0.44962433 -0.23139814 -0.03973598 0.15911146 -0.44687304 0.38062218 -0.41870263 0.020045359 -0.28846094 -0.0043901796 0.026963282 0.39557004 0.36582246 0.17082286 -0.6019602 0.14630833 -0.08861996 0.44384888 -0.16790947 -0.07841002 -0.041997727 0.08568078 0.271272 -0.075097695 0.3537214 0.048118737 -0.29863128 -0.48582652 -0.053672902 0.113357976 -0.22605768 -0.2837908 0.04309148 0.41172996 0.25012475 -0.17102103 -0.028994577 -0.0595139 -0.46165663 -0.24824232 0.17670767 0.51529276 0.3255087 0.40242448 -0.36754522 0.026979936 -0.20611316 0.21346067 0.37238127 -0.25270122 -0.15943308 0.66740286; -0.14951904 -0.20908634 0.2089761 -0.051702604 -0.16353223 -0.24620035 0.10563394 -0.05366431 0.30912113 -0.07354307 0.082265764 0.34062788 -0.42485046 -0.1949391 -0.1614228 0.16036318 -0.32441962 0.18401077 -0.49132892 -0.056715313 -0.046846416 -0.43271002 0.25954756 0.13639595 -0.72039324 -0.4489333 -0.2666304 0.24724893 0.0050859977 -0.4244259 0.037403904 0.29568756 -0.22635977 0.19542223 0.014404358 0.10868597 0.10903524 0.03409255 0.30537534 0.52094215 0.15022126 0.1856651 -0.36835703 0.24978341 0.2505714 0.23343623 -0.07613309 0.13015246 0.27363667 0.29085004 0.3087218 -0.012907274 0.28360367 -0.1174901 -0.40191054 -0.075704716 0.04647617 0.37567705 -0.062821046 0.2195001 -0.15221073 0.30467215 0.2665345 0.07979149 0.21110001 -0.10299078 -0.022880474 0.0148003455 0.2844238 0.29482523 0.012095819 0.32120624 -0.14038652 0.028053636 0.25232565 0.042305768 0.011736663 -0.4026612 0.22337128 0.45385867; -0.19405234 -0.22594844 0.2115495 -0.35722876 -0.53074867 0.22179501 0.0008540604 -0.28502217 -0.4063411 -0.39423686 0.5437821 0.5659688 0.016572807 0.1138164 0.2314023 -0.30498368 -0.33990186 0.28064573 -0.40707812 0.15236853 0.43972433 -0.13657433 -0.23223472 -0.043451607 -0.40156588 -0.5919947 -0.21389247 0.08975702 -0.5613461 -0.024061209 -0.09107399 0.04930415 -0.6340711 0.646463 -0.39390597 0.019283408 -0.587826 -0.17646988 -0.25648105 0.5089006 0.5552516 0.31796366 -0.5486169 0.25158912 -0.2095953 0.17065008 -0.19962533 -0.1481113 0.0014563894 0.28525057 -0.083494365 -0.20334579 0.40233994 0.2671476 -0.38702795 -0.51012576 0.14388731 0.3643816 0.121653125 -0.10505566 0.31901243 0.3299261 0.3878134 -0.4702187 -0.14108916 0.33290046 -0.47535115 -0.24619345 0.17890105 0.39128506 0.5630005 0.37736058 -0.22914371 -0.019468889 -0.33838823 -0.034252524 0.5844951 -0.5152633 -0.088350184 0.665252], bias = Float32[-0.13032989; -0.28966972; -0.13756396; 0.042905495; -0.2961647; 0.086586766; -0.04411684; -0.11181793; 0.26724735; 0.103950284; -0.13719517; -0.14018707; 0.20725836; 0.04285382; -0.23462379; 0.11201859; 0.20816502; 0.38888162; -0.19085792; 0.11979479; 0.19448873; 0.24682643; 0.07950181; -0.2889053; -0.13116689; 0.031155588; 0.36888546; 0.35584345; 0.17849843; 0.38685697; -0.2893773; 0.17915061; 0.09460933; 0.15984862; 0.021685963; 0.2586123; -0.2621961; 0.05805723; 0.29853734; -0.18962522; -0.091645956; 0.12760848; 0.22631046; 0.09760226; -0.30397397; 0.05741793; -0.088863604; -0.22189584; 0.003959421; 0.1468025; -0.081784114; -0.03614727; -0.20427255; -0.061445966; 0.14029373; -0.21768562; -0.013150413; -0.045529637; 0.2579944; 0.17268422; -0.08442857; -0.0060593695; 0.0014062888; -0.04283831; 0.027255254; 0.28186622; 0.10380893; 0.2758429; 0.001723795; -0.1292029; 0.10927276; -0.09983487; -0.22837266; -0.16139157; -0.2593886; 0.0028863936; -0.086842746; 0.010456207; 0.17868413; 0.12219586]), layer_3 = (weight = Float32[-0.20669049 0.6139451 0.19422026 0.28296998 0.42926857 -0.11317396 0.11344119 -0.33748168 -0.83589154 -0.7639057 -0.083844095 0.02030775 -1.1061504 -0.268204 0.5191573 0.15180029 0.38605356 -0.9318532 0.07715632 -0.056515712 -1.1844798 -0.14571482 0.17091021 0.10851514 1.3092034 -0.29175812 -0.8353329 -0.6163581 -0.34267092 -0.074865334 0.9657653 -0.1487772 -0.58025014 1.4229808 0.14522451 0.2591271 1.0544556 -0.6003693 -0.8513626 0.4449282 0.042215664 -0.11929594 -0.22461429 -0.30143493 0.943068 0.11037512 0.5584568 0.56737375 0.008385586 -0.42519802 0.5077476 1.1765869 0.08818101 0.61055654 0.19813095 0.16919422 -0.027571391 -0.0858394 -1.0402632 -0.20798361 -1.2608188 0.14067325 -1.2507223 -0.021062136 -0.40508574 0.082961634 0.024548326 -0.9400145 0.64147127 0.29212657 0.13383737 0.017354544 0.9968728 0.019860826 0.7185538 -0.013776711 0.22369657 0.06047803 0.49869323 -0.21775399; 0.022288201 0.98152804 0.058686595 0.17289914 0.44413337 -0.10542478 0.13963704 -0.16798645 -0.75512326 -0.6436645 0.07119554 0.094543874 -0.83688384 -0.42108276 0.47059795 0.07604921 -0.0052412907 -0.80023456 0.0007198646 -0.029123634 -0.6485455 -0.48704788 -0.040479835 0.46252736 0.98263836 0.021405043 -0.865593 -0.8756025 -0.31573376 -0.044644 0.6744475 -0.12275488 -0.5193442 0.8247248 0.10660428 -0.058147475 0.65276283 -0.5805468 -0.8851607 0.43244395 0.01674106 -0.13761137 -0.31257704 -0.15784955 0.8273041 0.0064854277 0.5231564 0.4786946 -0.15770875 -0.47555056 0.48387262 0.92044383 0.18542098 0.552483 0.056048207 0.29125705 -0.13376936 -0.07208617 -0.8450229 -0.45782846 -0.8883139 0.022538131 -0.9083644 0.006485233 -0.4236204 -0.35608363 0.20120381 -0.8074406 0.55982554 0.5004926 0.018538011 0.03704046 0.7755612 0.15112928 0.5737199 0.07112561 0.22859946 -0.104650065 0.16834657 -0.5666135], bias = Float32[0.067911565; 0.32790208]))

loss_neuralode(p_trained_moderate)


#High noise

rng = Random.default_rng()
Random.seed!(99)

#Adding moderate noise to data:

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 35e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))


#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)





prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-80]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained_high = (layer_1 = (weight = Float32[0.7306027 -0.6962298; 0.2203602 -0.5411892; -0.004561128 0.87172776; -0.104952835 -0.6419633; -0.03289624 -0.91529167; 0.66259336 0.6616381; -0.07601315 -0.10656825; 0.10797377 0.1903619; -0.18052684 -0.7291776; -0.6631399 -0.83202463; 0.5435769 0.8040849; -0.26046234 1.1486449; -0.32351404 -0.4385445; 0.11825486 0.34765038; 0.39418507 0.7345042; -0.40174377 -0.5816122; -0.1612657 -0.7036029; -0.14298888 0.7654527; -0.041051555 -1.350873; 0.1380499 0.296499; 0.1654029 1.0133158; 0.46172848 -0.93321496; -0.37058964 -0.28730905; -0.40979344 -0.5341042; -0.3729415 -0.5237222; -0.5409169 -0.60406786; 0.57115686 -0.7161477; -0.23251033 0.04494886; -0.24697024 -1.160943; 0.50903916 -0.69751596; -0.7860996 -0.4103016; -0.3316178 -0.5068643; 0.18869758 -1.0316395; -0.018916324 1.0888708; 0.13407592 -0.5587292; -0.5460813 -0.4792179; -0.55721366 -0.7909843; -0.8286036 -0.25197595; -0.15767014 -0.22749162; -0.24821322 1.0024865; -0.1578373 0.7613387; 0.31027883 0.5924922; 0.07659305 -1.1411297; -0.13710164 0.38284868; -0.07822104 -0.3507365; -0.26647732 0.5296945; 0.30147338 0.4873731; 0.24412236 -0.27061918; -0.64932823 -0.15965849; -0.4723289 -0.3225644; -0.075126104 0.006260883; -0.10198278 -0.083577245; -0.35282862 0.58421427; 0.10879757 0.36073545; 0.5544933 -0.8585583; -0.35235557 -1.1556267; -0.7496122 0.44676214; -0.57676095 0.95073783; 0.35960993 0.5121277; -0.06269343 -0.4232469; 0.21536714 0.66116166; 0.034315184 0.84665847; -0.35318336 1.0360743; -0.39294142 -0.68128806; 0.5079303 -0.9655373; 0.51461697 0.5154927; -0.496581 -0.896894; -0.20610197 -0.84791064; -0.69546217 0.5296343; 0.32910046 1.1421638; -0.21926904 0.71394736; -0.13495822 1.0468292; 0.8541351 -0.7636608; -0.12210916 -0.27302805; -0.5795929 -0.5827883; -0.38036108 0.45551684; 0.5723724 1.0454572; 0.13581304 -0.7371344; -0.17656502 -0.69315326; 0.2966321 1.1556077], bias = Float32[0.3267177; 0.18064685; -0.11676933; 0.3203592; 0.2743322; -0.20517716; 0.64938504; -0.5854452; 0.27990198; -0.26195285; -0.1391343; 0.15653099; 0.26982406; -0.6067975; -0.22771937; 0.2493943; 0.040078364; -0.22059648; -0.40977687; -0.61754084; -0.12247914; -0.0066342354; 0.39346477; 0.22318944; 0.22360232; -0.3379812; 0.39844805; 0.5619149; -0.04893225; 0.3946873; 0.017750166; 0.27885404; -0.16756266; -0.020053301; 0.53614014; 0.16420124; 0.06984037; 0.106915705; 0.5998283; -0.050418764; -0.3297865; -0.28529048; -0.08066768; -0.61634016; 0.6106619; -0.24343033; -0.30640468; 0.78215003; 0.09929176; 0.22052716; 0.70048285; 0.62027323; -0.36916384; -0.59160703; 0.26450402; -0.079516426; -0.62740225; -0.20067625; -0.26292074; 0.6150328; -0.4132412; 0.0545194; -0.052747883; 0.27380377; -0.2619098; -0.24022022; 0.07547737; 0.16193224; -0.45117044; 0.3586255; -0.41421953; 0.027909236; 0.41186696; 0.58684367; 0.17582175; -0.4626234; -0.020388976; 0.35341045; 0.279367; 0.222906]), layer_2 = (weight = Float32[0.66068184 0.7564289 -0.8443112 0.58203894 0.80583465 0.36050937 -0.4959436 0.33470088 -0.22446726 0.84901214 -0.07877865 -0.7406348 -0.52370596 0.18547297 0.13536073 -0.20564388 0.71706235 -0.7244361 0.646745 0.14455742 -0.49552312 0.5868904 -0.30221844 -0.23545435 -0.54562634 0.7778926 0.5746985 -0.5070159 0.8214719 0.6362863 -0.31104836 -0.26887432 0.8470647 -0.8092272 0.5085954 -0.33369684 -0.22878557 -0.512506 -0.1874993 -0.9170809 -0.90585494 -0.5540597 0.6425832 -0.349526 -0.05332079 -0.90524185 0.34155023 -0.30535126 -0.30448946 -0.56964666 -0.5623413 0.03109729 -0.77912563 0.10648222 0.7778015 0.86556417 -0.47754672 -0.7563988 0.35846946 -0.1740332 0.08877023 -0.92014515 -0.9447363 -0.3272338 0.15582852 0.1920391 0.34829792 -0.19342062 -0.569918 -0.5776869 -0.84413785 -1.0235645 0.63370746 -0.4579987 -0.24357685 -0.84130853 -0.8557197 0.7580799 -0.25627986 -0.7183136; -0.26840815 -0.33692318 -0.31724912 0.47933677 0.2740631 -0.69535506 0.7145987 -0.6890024 0.68930703 0.4708263 -0.9323563 0.10309216 0.6874338 -0.79909223 -0.76898146 0.9054286 0.4094932 -0.26619083 -0.33943653 -0.9584873 -0.6044821 -0.22995149 0.8012282 0.7643342 0.58214176 0.3035956 -0.34633195 0.59567934 0.64691156 -0.3719937 0.61561084 0.88581556 0.14895323 -0.59897274 0.6421737 0.6583339 0.92775893 0.6824186 0.7617194 -0.038005777 -0.25734237 -0.4241899 -0.13130714 -0.30669698 0.801492 -0.046936 -0.7254484 0.8929305 0.7163492 0.5892871 0.58486485 0.7468965 -0.18808384 -0.7879838 -0.6113455 0.40747237 -0.034484588 0.43616527 -0.9564557 0.8656797 -1.0065279 -0.17886645 -0.020037673 0.92379856 -0.23852526 -0.9273003 0.9197399 0.99276733 0.4153586 -0.0032211915 -0.5927603 -0.32964227 -0.5692432 0.6896083 0.90071297 -0.16569777 -0.531197 0.5077942 0.7732816 -0.17247705; -0.5005081 -0.24333005 0.7959267 -0.5387483 -0.7053336 -0.088323 0.6382256 -0.5526217 0.19597259 -0.71695256 0.41059452 0.76379496 0.3727643 -0.50955385 -0.12925711 0.30909678 -0.6902627 0.5155407 -0.69899315 -0.57354397 0.74777585 -0.71027637 0.45694038 0.45257336 0.28718305 -0.79771525 -0.71413463 0.58710515 -1.1128777 -0.51693887 0.12743497 0.26611683 -0.62210345 0.7438901 -0.41657883 0.20092312 -0.22951986 0.21041022 0.3937454 0.7497225 0.87842077 0.52405673 -0.73104966 -0.47201785 0.27098617 0.29525873 -0.55050695 0.70167214 0.21368106 0.29242256 0.43719813 0.5084643 0.4303398 -0.5772315 -0.82694596 -0.96113455 -0.28959978 0.7069713 -0.35520604 0.30924803 -0.117230974 0.8225678 0.82521397 0.07001302 -0.6427524 -0.31150225 -1.0179365 -0.23079567 0.047226164 0.73612916 0.76618963 0.62508225 -0.40441054 0.5260724 0.12783766 -0.31674019 1.117358 -0.747201 0.35551396 1.0446532; 0.72157484 0.7631729 -0.66505545 1.0841502 0.8061968 0.44624162 -0.47207925 0.12942792 -0.6497222 0.9187171 0.29215744 -0.583402 -0.34620175 0.20014353 0.3169615 -0.43656245 1.0151595 -0.8835675 0.6879683 0.26550195 -0.10028924 0.71342736 -0.384409 -0.5329535 -0.53561115 0.76127005 0.8660312 -0.1313214 0.508302 0.8652783 -0.4541375 -0.5408663 0.70646375 -0.6800416 0.5997828 -0.5572773 -0.3293295 -0.43071723 -0.22087862 -0.7149743 -0.9790382 -0.6189832 0.43038017 -0.5943271 -0.06628164 -0.6788264 0.6036642 0.046904653 -0.34443843 -0.53934306 -0.18590924 -0.1400754 -0.9459849 -0.0052659805 0.6578705 0.637041 -0.71092004 -0.61323535 0.3292458 -0.014009054 0.28148916 -0.67552054 -0.50063413 -0.33661827 -0.34928277 0.50205976 -0.48329988 -0.59955114 -0.7376309 -0.6924862 -0.9341266 -0.50438935 0.588536 -0.28781706 -0.47423282 -0.92316 0.030229913 0.87429047 -0.38351873 -0.5762764; -0.42294225 -0.36139128 0.59200174 -0.7437418 -0.8008739 -0.37283793 0.3824117 -0.47575969 0.39042163 -0.6522719 -0.24155904 0.51564634 0.6291595 -0.45731598 -0.41692814 0.2866054 -0.5791952 0.7654711 -0.777979 -0.5166349 1.1684086 -0.5725512 0.5038852 0.5685934 0.46490857 -0.5837989 -0.51163715 0.40075147 -0.6472039 -0.64464027 0.5075273 0.40900323 -0.67384344 0.8729558 -0.392085 0.4780855 0.2648152 0.4889173 0.28365916 0.7510038 0.6773294 0.3416668 -0.87498355 -0.46075177 0.49345106 0.43117374 -0.29633942 0.604212 0.46436924 0.6129445 0.6562277 0.3084855 0.6976982 -0.40822572 -0.70519626 -0.5846002 0.3960452 0.8353446 -0.3775307 0.40327346 -0.5433894 0.7856522 0.6802255 0.48213643 -0.70914936 -0.4365371 0.08448506 0.10357852 0.6044083 0.6343664 0.7444952 0.7644561 -0.5089985 0.6576961 0.45937705 0.19089186 0.856083 -0.697925 0.41167575 0.9144903; -0.5321335 -0.8876574 0.7293832 -0.7013886 -0.999768 -0.13126652 0.42696196 -0.2544735 0.55702233 -0.76282245 -0.0636043 0.8080545 0.40801698 -0.1105593 -0.13952586 0.46135125 -0.8560865 0.92084795 -0.52820194 -0.22531703 0.5810245 -0.6782068 0.017972643 0.46667552 0.38379517 -1.0301774 -0.8876936 0.124534704 -0.445643 -0.563767 0.44614866 0.4358738 -0.6035708 0.77128327 -0.8822489 0.55916715 0.15491681 0.25519913 -0.03685284 0.8468106 0.90408164 0.7184045 -0.77099407 0.6247504 -0.105905935 0.6874512 -0.3418332 -0.054299507 0.48408163 0.6032772 0.15124084 -0.23359236 0.8617994 0.007938601 -0.57564384 -0.5101658 0.5521918 0.65717804 -0.5629992 -0.005290153 -0.42548376 0.80162364 0.674715 0.39153785 0.28862476 -0.3061859 0.2062162 0.5126947 0.7425892 0.47916242 0.7323447 0.561627 -0.38075638 0.08645027 0.30978236 0.8720293 0.60724324 -0.9701897 0.23964319 0.66050684; 0.62191755 0.21895185 -0.76644206 0.5195658 0.79495734 -0.64453626 -0.06459251 0.41040164 0.7703162 0.66210014 -0.6081847 -1.1744641 -0.32558528 -0.24010476 -0.73878396 0.012963523 0.37420702 -0.8490565 0.78719187 -0.19908458 -0.8479347 1.1583954 0.09266606 0.08789046 0.15067117 0.78279895 0.61032254 -0.31128213 0.9495704 0.69197595 -0.36582184 -0.08159737 0.93707883 -1.0584017 0.5299647 -0.14679211 0.81767553 -0.15980741 0.1951353 -0.88439953 -0.6349815 -0.6068412 1.1114268 -0.21572107 0.44940427 -0.64579 -0.07994602 -0.13868807 -0.037402526 -0.42945752 -0.26786458 0.34942594 -0.5970165 -0.37926665 0.72765833 0.7333441 0.26215348 -0.7379619 -0.0191026 0.4251456 -0.66119677 -0.555934 -1.1423804 0.5071333 0.66332346 -0.29903045 0.69211936 1.2047765 -0.08397842 -1.0514151 -0.98592454 -0.9455714 0.5095371 -0.28258288 0.33170125 0.09774578 -0.9750584 0.8126531 0.74607474 -1.0323071; 0.66245234 0.58630097 -0.87187004 0.801841 0.79982275 0.23319554 -0.50628847 0.435452 -0.4925567 0.6907185 -0.15377493 -0.8536428 -0.49707678 0.2753309 0.057434656 -0.23328681 0.7204133 -0.68192464 0.59175915 0.37746352 -0.58600324 0.6399204 -0.16882236 -0.2330634 -0.5087338 0.7903777 0.7819981 -0.45629945 0.6562415 0.6647121 -0.62817913 -0.2683219 0.96210533 -1.0038015 0.6395105 -0.27848798 -0.06785622 -0.606294 -0.16131704 -0.7802779 -0.67802197 -0.76108396 0.78126794 -0.36699617 -0.061766066 -0.6024648 0.27363533 -0.27325848 -0.25151333 -0.59726137 -0.5174248 -0.28144222 -0.7117487 0.39032656 0.7907579 0.7739129 -0.5785657 -0.7613563 0.33140898 -0.055826742 0.07439412 -0.9083636 -0.8508265 -0.3865334 0.24698935 0.2117595 0.18648791 -0.16090432 -0.70342636 -0.61541766 -0.91660196 -0.6310756 0.5271158 -0.25102472 -0.20901966 -0.95485866 -0.7212801 0.735171 -0.16943496 -0.7330079; 0.34895673 0.19762254 0.0076334393 -0.11651031 -0.22987829 0.9398539 -0.6383073 0.3454541 -0.49660194 -0.5997455 0.6939895 -0.74571645 -0.14042473 0.40905425 0.41608223 -0.6770487 -0.044458356 0.021224923 0.51674205 0.42357847 0.17828374 0.79750913 -0.6996459 -0.8514636 -0.4770727 -0.7456074 0.5826817 -0.32712922 -0.20491078 0.52545536 -0.7227191 -0.9888394 0.50009125 -0.049436998 -0.06918882 -0.9306217 -0.4707779 -0.7653385 -0.42382723 -0.56129146 0.03060491 0.5058081 0.40441817 0.1394959 -0.6349928 -0.2720172 0.7061323 -0.22865476 -0.87354743 -0.802837 -0.4996742 -0.37716058 0.105011426 0.39543158 1.0598745 -0.36812335 -0.32233652 -1.0837007 0.8702115 -0.5064218 0.6664034 -0.024882976 -0.7570214 -0.5705931 0.43972602 0.974696 -0.34018502 0.18455128 -0.22672404 -0.2682101 0.18508798 -0.22396828 0.6524928 -0.53399265 -0.95096785 0.05735377 0.50406265 -0.32383135 -0.41932517 0.05647575; 0.6281255 0.42844418 0.14565332 -0.29419917 -0.6136383 1.0170408 -0.7269518 0.21280886 -0.6113442 -0.36010754 0.92078865 -0.49012953 -0.38002226 0.6223296 0.6058316 -0.8270551 -0.059520673 -0.0365065 0.57562244 0.5608291 0.62913597 0.6945299 -0.7198028 -0.94622636 -0.5418618 -0.7475615 0.71249324 -0.45061338 -0.7507849 0.3305808 -0.5683599 -0.97217333 -0.029287834 0.5753783 -0.16562335 -0.82472956 -0.7604647 -0.86181366 -0.6070231 -0.40346986 0.27963975 0.37848195 0.04679512 -0.09776569 -0.7134735 -0.27439705 0.59336746 -0.25565296 -0.8025049 -0.8056517 -0.7514488 -0.311325 -0.15704314 0.5952645 1.0928346 -0.60174805 -0.3762527 -0.97107375 0.95709485 -0.6685461 1.0149996 -0.29493925 -0.5794197 -0.7731379 -0.001968296 0.8734334 -0.7971425 -0.6504447 -0.37644196 0.120782964 0.21201415 0.13321042 0.8578495 -0.5428572 -0.86350465 -0.3209724 0.94873846 -0.4095849 -0.8676723 0.12656774; 0.5194392 0.77715945 -0.7071503 0.73461086 0.62308764 -0.19260873 -0.19632162 0.24498408 0.09167505 0.545132 -0.455776 -0.6605112 -0.24379067 -0.21235083 -0.36279154 0.008944645 0.888774 -0.6745451 0.6173542 -0.13950098 -0.6887416 0.86619705 0.016450701 -0.095112264 -0.15183854 0.6050539 0.5893559 -0.46054095 0.7549333 0.65228224 -0.2923615 -0.18653978 1.0573876 -0.68477356 0.53836024 -0.16386065 0.17018037 -0.25742006 0.1120719 -0.7470163 -0.83543056 -0.7170416 1.0042473 -0.51778644 0.2655827 -0.67508894 0.104657024 -0.114175715 -0.1832738 -0.3412902 -0.32221854 0.242722 -0.7001258 -0.32308558 0.64443624 0.91125983 -0.59535694 -0.69210607 0.3046522 0.24920951 -0.24286194 -0.7366944 -0.7391452 0.24377893 0.045944758 -0.121744275 0.5780933 0.40029293 -0.69813734 -0.9718815 -0.7249627 -0.67593926 0.5379041 -0.22866397 -0.18044229 -0.6388112 -0.7418917 0.73796386 0.08520813 -0.6678691; 0.75448644 0.53497815 -0.91354537 0.6316425 0.7360136 -0.40538687 -0.09123993 0.18216935 0.015660524 0.5470102 -0.5022711 -0.73535424 -0.48864767 -0.20649825 -0.54271847 -0.116127275 0.83575463 -0.6824822 0.81045735 -0.27478272 -0.6545531 0.56048775 0.2698839 -0.1874154 -0.18732226 0.57125866 0.8435074 -0.41727376 0.68405664 0.7870439 -0.29362905 -0.13195176 0.70183975 -0.8641308 0.6647888 -0.12289619 0.21393637 -0.15021166 0.16865532 -0.86548173 -0.6721869 -0.71917766 0.7360329 -0.42179513 0.41810352 -0.626333 0.22181638 0.32568312 -0.074163035 -0.43904433 -0.2715748 0.21910265 -0.55697745 -0.40175018 0.55470186 0.7095941 -0.5978715 -0.6034998 0.2324907 0.23994045 -0.26722538 -0.80897 -0.9740401 0.09244041 -0.030020734 -0.23748338 0.41781986 0.4120972 -0.3816796 -0.6243995 -0.55691016 -0.8220382 0.37424305 -0.13108227 0.17452691 -0.6293508 -0.81570596 0.78405166 0.2137754 -0.7368276; 0.78024995 0.50057966 -0.25419447 -0.08756455 0.24787736 0.7760657 -0.74468106 0.45105317 -0.4169803 -0.31035566 0.57574505 -0.7228105 -0.5645234 0.6634993 0.582049 -0.79153645 0.056268312 -0.2998234 0.51716805 0.6252448 0.3205951 0.7704951 -0.5785368 -0.6476544 -0.53742397 -0.23816991 0.5095591 -0.40425122 -0.018195067 0.34410673 -0.5196729 -0.8052981 0.8850238 -0.0041941237 -0.29613942 -0.8264905 -0.6842909 -0.7641147 -0.6928008 -0.73657095 -0.3472315 0.4130691 0.6118189 0.15882383 -0.44489637 -0.17967032 0.7376394 -0.50073373 -0.9677872 -0.76209134 -0.573954 -0.63135064 -0.27054408 0.42706493 1.075258 -0.066458926 -0.08447795 -0.8427672 0.7529622 -0.5281015 0.90657085 -0.3619135 -0.74929076 -0.6151454 0.33694008 0.8853873 -0.4157133 -0.46532792 -0.382645 -0.17972848 -0.035333 -0.53395873 0.93288213 -0.3760904 -0.85902154 -0.22556663 0.40760958 0.056665327 -0.4547313 -0.27604914; -0.24113257 -0.26546782 0.57626694 -0.46835423 -1.0300347 0.62982297 -0.7149335 0.49476564 -1.1420052 -0.4828607 0.616867 1.1589148 -1.0101022 0.64785737 0.97495127 -0.76643467 -0.63140064 0.78560406 -0.4384707 0.49940947 1.0156296 -0.61023396 -0.76510274 -0.4684999 -0.91909695 -0.3331383 -0.08791906 -0.34161365 -0.7003689 -0.24834648 -0.17499371 -0.6060204 -0.9267134 1.1803902 -0.6200726 -0.2630233 -0.3702712 -0.40730372 -0.7926066 1.0268786 0.80163515 0.7751381 -1.1584849 0.33321986 -0.5947347 0.10713066 1.1559938 -0.4159382 -0.3346826 -0.6357752 -0.3416366 -0.59822094 0.3371218 0.64329153 -0.3148738 -0.67097306 0.1114165 0.3628677 0.6109842 -0.77227855 0.8562417 0.6143658 1.0476863 -0.88946396 0.11630894 0.53749883 -0.543247 -1.4159212 0.16464289 0.6235457 0.8369357 1.0043261 0.20415176 -0.469696 -0.6105559 0.25508043 0.6397367 -0.6563019 -1.1106404 0.33370903; -0.69527066 -0.46965045 -0.2724071 0.23076981 0.29379466 -0.89095235 0.53506607 -0.12086998 0.7361556 0.37159353 -0.9102277 0.41330284 0.040914092 -0.41689453 -0.59493387 0.99218434 -0.118147634 -0.034893382 -0.25818545 -0.37300092 -0.6209935 -0.58590263 0.7022115 1.1857185 0.59589577 0.9466602 -0.54242676 0.27854335 0.62564015 -0.35308403 0.9197324 0.95533985 -0.19323455 -0.51902753 0.23968229 1.0475044 0.9524649 1.0999205 0.40439996 0.0015075 -0.1900608 -0.5228284 0.3195655 -0.101113066 0.33313137 0.34034783 -0.7269666 -0.079560556 1.1417402 0.9154083 0.32325733 0.21395664 -0.15173522 -0.42949942 -0.93112016 0.63806134 0.514513 1.1252968 -1.0313605 0.1861586 -0.89455646 0.059065513 0.43008828 0.8820489 -0.009863268 -0.8347594 0.7073984 0.630155 0.5704024 -0.28206235 -0.19234657 0.025434142 -0.96053785 0.24921097 1.0948112 0.21648937 -0.90771884 0.07463959 0.69033104 -0.5515609; -0.71090394 -0.5044706 0.73373806 -0.6352767 -0.9329006 -0.25207806 0.564291 -0.3476683 0.39360306 -0.9936445 -0.23344094 0.8655192 0.29337144 -0.41006416 -0.06380218 0.35682288 -0.991386 0.6335237 -0.5066069 -0.2632027 0.658043 -0.85615355 0.46764126 0.21260813 0.27202573 -0.6763213 -0.6615767 0.26393324 -0.87536514 -0.6166483 0.60782796 0.3488291 -0.76524276 0.99299973 -0.6863173 0.5328538 0.20950724 0.41786745 0.4345312 0.7374794 0.7072881 0.5957425 -0.7376885 0.6509053 0.050495796 0.8351941 -0.31332606 0.4567094 0.566707 0.42032164 0.5715794 0.14964429 0.9988395 -0.08046836 -0.7965196 -0.9036072 0.8323553 0.66726404 -0.28018034 0.12762153 -0.38972217 0.8109793 0.5708773 0.4434238 -0.39651296 -0.3668603 -0.045562424 0.36031651 0.6652486 0.6568026 1.0016881 0.8534208 -0.75014836 0.24094313 0.44989774 0.64138234 0.5991347 -0.8917002 0.18269795 0.5921638; -0.5092583 -0.81925565 0.7614385 -0.656399 -0.67134285 -0.2021242 0.29194131 -0.59241265 0.42267662 -0.59143245 0.12432794 0.9169539 0.22466904 -0.059183285 -0.008653175 0.28551114 -0.6729939 0.69660187 -0.52433515 -0.05381222 0.4853836 -0.55077344 0.13684072 0.21351813 0.46503118 -0.7119285 -0.60253966 0.33110583 -0.6504572 -0.77517694 0.35606188 0.4560745 -0.9612273 0.6150106 -0.43913263 0.21979013 0.26764268 0.48786244 0.06506977 0.84927875 0.8678797 0.76218766 -0.91531235 0.7455414 -0.0028192932 0.7717662 -0.26859352 0.26662204 0.18386662 0.38757315 0.37859643 -0.097555175 0.65990484 -0.15316494 -0.514955 -0.7882781 0.7999621 0.5484043 -0.3948117 0.09108469 -0.0077105835 0.69674265 0.96613574 0.0808648 -0.24332075 -0.22691205 -0.20725878 -0.03022421 0.57226825 0.8288334 0.7376432 0.8660218 -0.53865314 0.37616035 0.115841165 0.88507724 0.71288455 -0.8214546 0.3163905 0.697253; 0.5956294 0.2993575 -0.2661648 -0.2616123 0.11361771 0.72229725 -0.75515157 0.29251763 -0.42997214 -0.2823679 0.7396293 -0.7756214 -0.40728787 0.59541094 0.5188928 -0.7073571 0.22526334 -0.2882552 0.4519979 0.56387496 0.33153462 0.8282602 -0.708905 -0.65017205 -0.51429975 -0.547889 0.5796037 -0.7209031 -0.09580757 0.49997634 -0.78943986 -0.6240183 0.5382763 0.15716912 -0.40554872 -0.7135532 -0.51155716 -0.8474866 -0.6077275 -0.71626645 -0.20928542 0.32011345 0.3413403 0.20200182 -0.52285886 -0.33466694 0.6111381 -0.44677457 -0.8563591 -0.8897999 -0.6436371 -0.6203899 -0.20002235 0.5829246 1.025666 0.13280278 -0.37167457 -1.1565751 0.71894825 -0.622374 0.7738903 -0.29796484 -0.79470885 -0.5880343 0.23192514 0.78685474 -0.72066736 -0.24213868 -0.44307354 -0.16174261 0.088797264 -0.57801765 0.86862326 -0.50944227 -0.6014968 -0.16486365 0.46925178 0.08764289 -0.6622156 -0.28801024; -0.72181875 -0.56400156 0.80329376 -0.7549822 -0.85970205 -0.0068442794 0.451693 -0.62161946 0.046726618 -0.54024315 0.32466063 0.69058615 0.29948953 -0.0976373 -0.066091344 0.2711783 -0.60890406 0.6944902 -0.5265068 -0.19459859 0.9706364 -0.544829 0.38227051 0.34989205 0.4761285 -0.7492308 -0.76283735 0.652784 -1.0171043 -0.70199186 0.30772737 0.2291871 -0.96727043 1.1269027 -0.3739918 0.37979534 0.13231471 0.36178914 0.3743165 0.90908366 0.8405684 0.409589 -0.86553603 -0.07167596 -0.0094632525 0.7254626 -0.20595883 0.52927417 0.3023163 0.37493485 0.34367654 0.41828248 0.61074275 -0.2599713 -0.57089925 -0.8366641 0.067763396 0.534472 -0.48104677 0.017521916 -0.25478682 0.57784414 0.7875629 0.29991418 -0.7632611 -0.46969298 -0.62747574 -0.21888901 0.61722946 0.9060819 0.82060546 1.0603391 -0.6816525 0.4680621 0.07793607 0.14566347 1.0679522 -0.6949849 0.06413509 0.80887276; 0.4717028 0.663747 -0.57116985 0.7266917 0.7590426 0.3442271 -0.29845792 0.55279845 -0.24499197 0.73276514 0.42275277 -0.8471189 -0.4744179 0.24483672 0.18067473 -0.3482514 0.67108065 -0.90708977 0.48329163 0.42799076 -0.9645367 0.7483761 -0.551765 -0.3120678 -0.49323952 0.6437772 0.6354759 -0.63717055 0.84022075 0.8265876 -0.5238933 -0.2698291 0.8366579 -1.0186294 0.5850738 -0.4662921 -0.34147003 -0.4818828 -0.5320512 -0.85801995 -0.6728816 -0.38756326 0.79439276 -0.14082336 -0.35127556 -0.6198946 0.47238016 -0.3092104 -0.37532228 -0.26188046 -0.43470684 -0.3578782 -0.9075163 0.29144356 0.8362614 0.8262075 -0.6808934 -0.8456913 0.35064614 -0.27625924 0.22239748 -0.5497401 -0.8261318 -0.51361716 0.47057006 0.26493013 0.25902447 -0.11114313 -0.60447764 -0.8874841 -0.78011274 -0.9006108 0.8431549 -0.44311175 -0.5557513 -0.68159664 -0.8453454 0.6326841 -0.3921728 -0.6466888; 0.1843332 0.0060865153 0.01585115 -0.5505639 -1.6713305 0.26331446 -1.0077084 0.73208886 -0.58067095 0.28370246 0.00058635947 -0.7495601 -0.3072994 1.0202895 0.11320151 -0.28777266 -0.12722515 0.23375815 0.6396504 0.9264661 0.24392134 0.44915372 -1.0201805 -0.61867774 -0.34431255 0.3916281 -0.14158846 -1.0144835 0.30344218 -0.08419312 0.12964287 -0.7546723 0.26661313 -0.06203141 -1.0932302 -0.4112046 -0.17956918 -0.50270474 -0.9904421 -0.41150808 0.5734993 0.60647595 0.6794064 0.440702 -1.0469406 0.095678456 0.6760175 -0.8965299 -0.3372868 -0.53123933 -1.0022762 -1.0500628 0.20119087 1.1056715 0.54750025 0.3510377 0.33937275 -0.6343738 0.86376005 -0.9025218 0.76521623 0.16689123 -0.6902361 -0.3057534 0.4553722 0.4981976 0.21804632 -0.34176913 0.18663447 -1.0700283 1.0419489 -0.09605119 0.026935758 -0.67416924 -0.43487582 0.15803118 0.0068148864 -0.8812618 -0.5918922 -0.57757515; 0.7351006 0.5150476 0.40160266 -0.40328535 -0.4079243 0.81720954 -0.5744877 0.5778478 -0.97565305 -0.37170216 0.8329428 -0.1044783 -0.6087448 0.5533492 0.9679487 -0.8730584 -0.16261697 0.3578887 0.50530726 0.49317238 0.7515769 0.7865767 -0.7011582 -1.0741267 -0.856087 -0.7306898 0.41953167 -0.60272586 -0.7863643 0.44303852 -0.78556144 -0.9613879 0.13786198 0.672942 -0.2838999 -0.7792233 -1.0572053 -0.99718314 -0.52134985 0.09713054 0.16984361 0.32612425 -0.325216 0.018130383 -0.6199159 -0.19620943 0.7549411 -0.19834355 -0.7263315 -0.653319 -0.6643441 -0.44949782 0.18160772 0.61045814 0.95103276 -0.6313369 -0.693166 -0.7863232 0.8716619 -0.4241047 0.8902677 0.13258259 -0.3528669 -1.0506202 0.008146961 0.9167346 -0.9322322 -0.5977203 -0.717253 0.21361266 0.56506026 0.5247546 0.69097 -0.6497813 -0.9314297 -0.36371917 0.76794064 -0.44637963 -0.74190694 0.36313704; -0.65185225 -0.66310877 0.78907436 -0.65084094 -0.75423753 -0.28186107 0.4734222 -0.4713108 0.37466523 -0.62723404 -0.47278947 0.5961271 0.37601513 -0.38281894 -0.27737784 0.57573104 -0.7938612 0.8041934 -0.4826594 -0.41125244 0.7500222 -0.58838654 0.56356484 0.30660537 0.6100432 -0.7875334 -0.7263406 0.5800678 -0.7789821 -0.571692 0.3632859 0.26192346 -0.93742114 0.70530844 -0.49321803 0.5450119 0.50923 0.41496208 0.22322685 0.6991005 0.8077694 0.38615057 -0.6934201 0.4816652 0.47212997 0.5821197 -0.41207302 0.26159114 0.61894184 0.36886513 0.5914504 0.37153545 0.83524877 -0.34492943 -0.78785485 -0.7807895 0.5481343 0.6142285 -0.47927248 0.23189884 -0.2941371 0.65170455 0.75562304 0.53982043 -0.6197667 -0.4545495 0.054607723 0.15201718 0.48257577 0.62803155 0.94423765 0.84453034 -0.80355126 0.5447465 0.514506 0.5022731 0.8803497 -0.95269555 0.41652453 0.59391785; -0.69734967 -0.3582618 -0.09727289 0.21344651 0.48741108 -0.9188326 0.6002282 -0.42388573 0.8369909 0.68537027 -1.0282186 0.40004936 0.5465271 -0.8874419 -0.682443 1.0378349 0.0042217183 -0.038343992 -0.3988615 -0.52589625 -0.8200011 -0.8737065 0.6834649 0.69888645 0.5378367 0.7040979 -0.36966583 0.52473325 0.65647805 -0.7060657 0.71893847 0.9897141 -0.113338366 -0.3967345 0.2781229 0.74003726 0.9968412 0.7336454 0.68367106 0.27193174 -0.11556449 -0.5202562 0.059260514 -0.14421508 0.69856423 0.472292 -0.6125306 0.23299822 0.70887625 0.71151954 0.75330216 0.4435008 0.028436093 -0.4942531 -0.9433309 0.58141345 0.39861012 0.5930388 -0.6917037 0.41454503 -0.7163533 0.16230997 0.4610756 0.9019973 -0.086509444 -0.90577257 0.6692848 0.6489631 0.51789415 -0.12997207 -0.5232536 -0.3322797 -0.7576451 0.57274884 0.8352206 0.27703258 -0.83673644 0.24483155 0.9235246 -0.18857612; -0.43195942 -0.2797826 0.32100075 0.32066786 0.31195837 -0.7527895 1.0088995 -0.84979445 0.54588467 -0.3000392 -0.6487629 0.48531336 0.57564175 -1.0022492 -0.59370494 0.6858727 -0.34470132 0.06193983 -0.5300178 -0.99341834 -0.39774433 -0.5094814 0.7664182 0.8682709 0.45335934 -0.21741034 -0.17768526 0.6628723 -0.2130843 -0.20068629 0.5367513 0.6046336 -0.48735818 -0.13409209 0.6629927 0.6601782 0.56255794 0.6551491 0.7892957 0.45568135 -0.05329144 -0.65843683 -0.29643002 -0.40911362 1.0544894 -0.029456532 -0.60972255 0.6807085 0.8157133 0.8554589 0.91224843 0.8890997 -0.070663415 -0.8572397 -0.53953993 -0.14932966 0.019771935 0.5704359 -0.83418113 0.9978249 -0.7303261 0.46602556 0.46942762 0.75204515 -0.20841634 -0.72990835 0.20581114 0.34741116 0.15721612 0.72206277 -0.1622582 0.23320784 -0.7196142 0.8544613 0.54094386 -0.03129313 -0.06608321 0.2783697 0.6663652 0.6379786; -0.019365784 -0.33666945 -0.15397808 -0.10237862 0.7533628 -0.52639455 -0.7510496 0.94073343 0.123804845 0.78378534 -1.0398518 -0.79152024 -0.61673933 0.61328405 -0.9421734 -0.046578947 -0.068906374 0.0017257598 0.6028154 0.66863155 -1.2727442 0.56577986 -0.7955608 -0.1765609 -0.1994402 0.7968632 0.13237159 -1.016838 1.1487426 0.028419934 0.48519507 -0.39430714 0.9218864 -1.0713285 -0.38523987 0.12317803 0.93812495 0.086366974 -0.8782467 -1.0250734 -0.36264706 -0.06238926 0.9656984 0.71716994 -0.5102149 0.26422247 0.45829204 -0.9497648 -0.20576407 -0.48617998 -0.72460115 -0.79361707 0.17105378 0.8665052 0.6845996 1.3428613 0.7978398 -0.8049296 0.1515544 -0.48231044 0.020237679 -0.2338693 -0.73550385 0.29127526 0.8114552 -0.09094171 1.0410198 1.0474955 0.54837996 -1.3578922 0.07323166 -0.94332534 -0.022731986 -0.75122434 0.3077936 0.81958026 -1.3723243 0.2148178 -0.17770398 -1.3201066; 0.8132142 0.45279583 -0.19542636 -0.19766788 0.21419191 0.798502 -0.5658352 0.69357365 -0.42419705 -0.3288026 0.68338335 -0.69617957 -0.4854495 0.6971453 0.5128713 -0.75836176 0.14002597 -0.18641171 0.51119536 0.5621284 0.0042714993 0.83587146 -0.68932575 -0.6148233 -0.4608338 -0.29168975 0.52063215 -0.49906427 0.13401248 0.46084687 -0.5881045 -0.52353865 0.82832545 -0.53365064 -0.21823594 -0.62182605 -0.5835147 -0.8831419 -0.5651355 -0.72985286 -0.52210414 0.44542283 0.72869855 -0.05721715 -0.5256536 -0.3320093 0.6689632 -0.45859808 -0.8120324 -0.6354489 -0.53090453 -0.6105001 -0.39509726 0.46874532 0.94461375 0.31300306 -0.14471254 -0.9906711 0.5745815 -0.4222799 0.51160365 -0.4688839 -0.78753185 -0.6958244 0.9587528 0.6634158 -0.46328858 -0.2183615 -0.41723698 -0.440924 -0.24036773 -0.81413066 0.80812305 -0.48334923 -0.63827866 -0.16298336 0.17207617 0.40390635 -0.680532 -0.20884852; 0.36448944 -0.6381687 0.31700104 -0.47613826 -0.20340794 0.41562083 -0.81730515 0.6611427 -0.54523385 -0.48737463 0.39872453 -0.26138875 -0.5932646 0.61958724 0.5019106 -0.5844998 -0.6133774 0.49096882 0.007587724 0.5877684 0.20210208 0.49163085 -0.96478564 -0.8250463 -0.66978705 -0.25482094 0.3350739 -0.6235346 -0.15434559 0.20980245 -0.48708645 -0.8918778 0.33676258 0.32501706 -0.44676483 -0.87093204 -0.30590647 -0.74560034 -0.8512341 -0.03639356 0.51479876 0.5033425 -0.1172009 0.86789507 -0.57292897 0.2217477 0.81091803 -0.67789054 -0.8823497 -0.60296476 -0.6895931 -0.69537544 0.61533225 0.592202 0.72391236 -0.25579286 0.70359117 -0.60985416 0.9511149 -0.70564926 0.7179138 0.36824733 -0.019699613 -0.6959111 0.84456664 0.6183936 -0.11657001 -0.30429924 0.1550287 -0.14759003 0.4354419 0.058169905 0.62346786 -0.89627945 -0.60260385 0.8058225 0.4062094 -0.33975175 -0.73979074 -0.14550832; 0.46193042 0.6103466 -0.7707971 0.652776 0.56234944 0.19842306 -0.5250305 0.618058 -0.4277915 0.9643601 0.32151547 -0.8617223 -0.60125864 0.37391588 0.4439615 -0.3969698 0.7676379 -0.78538364 0.7642367 0.5007706 -0.8275746 0.49676564 -0.31840953 -0.3248835 -0.28063524 0.78668886 0.8079416 -0.2973705 0.82284707 0.642604 -0.7606516 -0.4583593 0.6598841 -1.0480556 0.53708297 -0.27630973 -0.4727503 -0.58319145 -0.20768762 -0.8531372 -0.6484524 -0.38933137 0.5970793 0.17997196 -0.28044078 -0.53994095 0.5024249 -0.5605058 -0.5113383 -0.27502808 -0.33587474 -0.3005117 -0.79002017 0.43413398 0.54160833 0.7120987 -0.42435655 -0.61669225 0.38472772 -0.2627134 0.547604 -0.69514346 -0.8310831 -0.4691443 0.4176889 0.3225044 0.035397254 -0.15286863 -0.5518331 -0.5821355 -0.928211 -0.68737817 0.44311467 -0.48969793 -0.41647494 -0.46987307 -0.9667623 0.9514613 -0.2287277 -0.80664366; 0.08469934 -0.24088137 -0.41361326 0.039977714 0.70139885 -0.22773741 -0.5264171 0.9418734 -0.30779722 0.8561962 -0.68383425 -0.82431775 -0.46223643 0.58420515 -0.044868156 -0.24298444 -0.029845892 -0.4674483 0.70407397 0.55546427 -1.1753474 0.6173169 -0.5272979 -0.48362175 -0.36050197 1.0611287 0.43532884 -0.6350871 1.1658814 0.24269126 -0.10025113 -0.24266924 0.91516405 -1.2390865 -0.3086382 -0.4188721 0.48210227 -0.08303207 -0.66663927 -0.94893444 -0.5766405 -0.011090936 0.7955673 0.7394786 -0.7122709 0.18809661 0.6003715 -0.87132967 -0.21329229 -0.3659492 -0.6743119 -0.6070901 0.11381746 0.8278295 0.73246586 1.1000845 0.7227803 -0.8339285 0.55098534 -0.47236055 0.1961044 -0.39709428 -0.65243727 -0.21510655 1.0400795 0.36918554 1.0349168 0.47015715 0.34634143 -1.0885773 -0.32770485 -0.9502351 0.30403224 -0.8226769 -0.18318798 0.6755067 -1.3350285 0.24759233 -0.41740227 -1.1737672; 0.04234769 0.23653533 0.33702573 0.6460268 -0.5469989 0.008730564 0.899036 -0.9437627 0.38219464 -0.49487334 0.396578 0.9260767 0.6549161 -0.9103339 -0.19277608 0.10476388 0.17461799 0.011410977 -0.51264143 -0.9503664 1.3648763 -0.69316137 0.6429197 0.41599706 0.51158834 -0.78484935 -0.1983729 0.8865792 -1.0768497 -0.23928557 -0.04097462 0.5368785 -0.8574236 1.2512153 0.95841867 0.31619853 -0.5637274 0.1367199 1.0244368 0.948123 -0.04188705 -0.6894414 -0.801193 -0.74361503 0.9056081 -0.35690582 -0.29805762 1.1279582 0.41541448 0.63263017 0.94719326 0.91836923 -0.39667 -0.8121554 -0.79128426 -1.3349854 -0.7944847 0.47791332 -0.2450341 1.2157077 -0.5045101 0.26392257 0.8034831 0.006558142 -0.9660197 -0.20553783 -0.8585075 -0.34430924 -0.3225011 1.171273 0.027995419 1.040547 -0.43400964 1.1050618 0.25972706 -0.5417029 1.2408123 0.09787346 0.3094214 1.0004303; 0.6106786 0.3949291 -0.81963754 0.72693765 0.8885486 0.48738164 -0.43119046 0.5776762 -0.26796466 0.74140173 0.40893134 -0.6138831 -0.4736511 0.48080567 0.48136523 -0.35477206 0.51645815 -0.7971117 0.79716414 0.28983128 -0.2863101 0.6955544 -0.4348964 -0.50587565 -0.61005735 0.58852935 0.699486 -0.51891196 0.95476097 0.81175226 -0.6554509 -0.31234086 0.6949881 -0.979131 -0.048046917 -0.4867467 -0.3711409 -0.49089977 -0.6301325 -0.9017059 -0.55193895 -0.06492473 0.8035825 -0.1415658 -0.48242736 -0.5267914 0.52973145 -0.67342615 -0.591802 -0.6171311 -0.28359562 -0.6521442 -0.71992177 0.6035718 0.75817126 0.69907594 -0.48077402 -0.8230236 0.33197445 -0.48641968 0.333195 -0.6451462 -0.8270965 -0.34675717 0.62305444 0.4995232 -0.42774907 -0.32976508 -0.5711263 -0.7356891 -0.87828165 -0.9076542 0.46091658 -0.51953936 -0.2825828 -0.3359609 -0.5702117 0.56896025 -0.21335807 -0.8154112; 0.69399726 0.4469871 -0.20098421 -0.4388084 -0.24626428 1.0360179 -0.756051 0.40554047 -0.78610295 -0.43521148 0.8974467 -0.5005331 -0.44031215 0.7923012 0.85019165 -0.58302355 -0.07405366 0.013908447 0.49757963 0.838368 0.95169735 0.5327561 -0.8127981 -0.9378502 -0.5504171 -0.5659321 0.58271307 -0.5656039 -0.7241557 0.52245176 -0.8873246 -0.87175876 0.11202059 0.4929725 -0.42810974 -0.6697007 -0.8423687 -0.8748801 -0.7960476 -0.32718596 -0.13653584 0.43547738 -0.047422603 0.06302882 -0.5541296 -0.24944532 0.832083 -0.23268606 -0.63705856 -0.5275943 -0.5425009 -0.5465228 -0.22635277 0.5741825 0.8807421 -0.30525035 -0.6101823 -0.735789 0.87167567 -0.5628191 0.75390095 -0.15094835 -0.5465117 -0.81926435 -0.066755325 0.65808624 -0.92088825 -0.79985183 -0.6429119 -0.054031532 0.3442845 -0.011079688 0.85464597 -0.6769317 -0.7310384 -0.22926049 0.9279937 -0.32580394 -0.62863266 0.18451293; -0.50371844 -0.8672654 0.55278367 -0.90148014 -0.71580297 -0.6025222 0.32111448 -0.563648 0.27073714 -0.55835885 -0.6455562 0.4859877 0.5495584 -0.32296517 -0.4576753 0.46826196 -0.6110622 0.6287972 -0.74706596 -0.585377 0.2584006 -0.6370674 0.5517657 0.2754001 0.24334005 -0.48091882 -0.6620653 0.5787833 -0.66680205 -0.68684024 0.8714868 0.5890743 -0.80313 0.58554065 -0.38652614 0.3646297 0.66430604 0.4290108 0.3387918 0.5409364 0.65043557 0.45687693 -0.7224666 0.46744704 0.5460411 0.6339868 -0.5225142 0.27366856 0.27411145 0.5240066 0.61936575 0.56042093 0.8807647 -0.46345425 -0.83858865 -0.5868035 0.9412012 0.6566682 -0.26614565 0.39832267 -0.3415092 0.5385086 0.8663408 0.28144228 -0.24993457 -0.3118828 0.36962092 0.53063715 0.7993235 0.6824103 0.6688618 0.79869723 -0.69578433 0.2570158 0.40164384 0.69829154 0.34173927 -0.57178754 0.37483165 0.4894635; -0.51643956 -0.6905693 0.86345845 -0.9255798 -0.91381633 -0.52267045 0.30757254 -0.6285391 0.26753396 -0.7071018 -0.27852377 0.75106704 0.28852355 -0.17910099 -0.48149076 0.45827684 -0.5954876 0.86132264 -0.48821187 -0.4586776 0.57767284 -0.7287906 0.29716873 0.592603 0.45296937 -0.657556 -0.74613166 0.58643866 -0.6983615 -0.5950628 0.49600178 0.35932526 -0.6662588 0.7261773 -0.43622264 0.29157916 0.22042175 0.49218214 0.24800038 0.62863874 0.8099032 0.3803519 -0.69600934 0.40173385 0.09074219 0.72804105 -0.46132302 0.27776232 0.6053873 0.33637473 0.49626032 0.47058052 0.63038296 -0.5497622 -0.73569125 -0.8789704 0.60173726 0.7281924 -0.2965603 0.32767427 -0.45841524 0.6122904 0.59642065 0.5507329 -0.39600733 -0.41076174 0.012508851 0.2985847 0.7688629 0.605083 0.88091457 0.8426166 -0.5604596 0.46894014 0.20916833 0.6787196 0.9287047 -0.82972854 0.21959922 0.69363636; -0.8087311 -0.6455069 0.6619391 -0.91794336 -0.9390779 -0.35158208 0.21552739 -0.22857527 0.2954496 -0.65330297 -0.034217924 0.8090081 0.6191802 -0.29587525 -0.27962387 0.5378776 -0.7660576 0.78807414 -0.63258153 -0.24662454 0.37749475 -0.6552806 0.11390226 0.4921323 0.25290048 -0.745009 -0.5916547 0.28017154 -0.94161093 -0.7773252 0.3001094 0.44521022 -0.9512988 0.6621949 -0.57189566 0.5469312 0.31387293 0.541789 -0.09124947 0.73551255 0.6923744 0.45881158 -0.7102118 0.6658972 -0.07920652 0.9183061 -0.21655104 0.21002726 0.34419453 0.28017768 0.28638244 -0.011645248 0.7945715 -0.22227965 -0.28769332 -0.68582904 0.6026944 0.77603126 -0.253668 0.032698322 -0.3134895 0.7600379 0.920119 0.16092816 0.15862378 -0.31836838 -0.27768865 0.12774819 0.5866238 0.71032304 0.8434087 0.78327996 -0.36403212 0.46511254 0.42789668 0.8200821 0.4413275 -0.8913536 0.3833026 0.7418268; -0.65164906 -0.31636256 0.6568271 -0.07361267 -0.5028997 -0.68925273 0.49294978 -0.62130356 0.26183948 0.019575141 -0.51456195 0.7342176 0.29301688 -0.7542037 -0.5908731 0.7195615 -0.3307709 0.2664435 -0.48983002 -0.4719235 0.37724718 -1.1042819 0.69188666 0.59888506 0.52576417 0.15706126 -0.41929656 0.81985044 -0.39808932 -0.70806825 0.48501796 0.52007943 -1.0405302 0.5813387 -0.017544225 0.71103764 0.588962 0.9758466 0.8198272 0.72762454 0.3875127 -0.22438532 -0.6923208 -0.14685431 0.4068239 0.44176188 -0.5052953 0.2781575 0.9452806 0.77090347 0.60919315 0.4491245 0.15479131 -0.73469156 -1.1189225 -0.39377537 0.3288816 0.8966274 -0.61076343 0.5906179 -0.5648455 0.27264038 0.6578058 0.42052227 -0.71362233 -0.7452383 0.1977163 0.044026617 0.7295672 0.3737899 0.5382796 0.72989523 -0.64362776 0.55626583 0.76503855 0.30066574 -0.0015034484 -0.45014706 0.29706484 0.53860855; 0.47067985 0.1083463 0.5610949 -0.5082428 -0.7004032 1.0130203 -0.6368677 0.41191718 -0.89092284 -0.64921737 1.053288 0.3606463 -0.645722 0.81560713 1.0615668 -0.958421 -0.33753684 0.551329 0.2656309 0.814023 0.6892352 0.06282429 -0.797315 -0.72744656 -0.89048934 -0.5328064 0.41015682 -0.7402696 -0.63946277 0.3608149 -0.68723834 -0.6922584 -0.7022095 0.71309054 -0.46457592 -0.80113864 -0.9598251 -0.6555453 -0.8784404 0.14016159 0.2558223 0.76065814 -0.4336267 0.23619668 -0.7269822 -0.13124067 0.5975092 -0.6213121 -0.626273 -0.51374555 -0.68451494 -0.78209734 0.23712267 0.65166295 0.23007333 -0.7843717 -0.3516018 -0.2788488 0.9047976 -0.68985724 0.9068874 0.41703078 0.04435412 -0.7886979 -0.13250381 0.7655143 -1.0488042 -1.0429842 -0.1489089 0.03709146 0.41993484 0.80732447 0.39464772 -0.60624415 -0.8254713 0.02976075 0.8612353 -0.42622903 -0.7252141 0.31249297; 0.28996265 -0.0716322 -0.052474286 -0.15560503 0.12643564 0.6394357 -0.75201803 0.64379 -0.8132122 0.64671564 0.3468999 -0.47900972 -0.41129673 0.92476904 0.6166753 -0.76054287 0.2686728 -0.24589117 0.58541286 0.95989347 0.19356216 0.59232926 -0.6368725 -0.58334404 -0.7112064 0.36315843 0.25606385 -0.7860894 0.2826986 0.09148713 -0.4113346 -0.64168483 0.659517 -0.2590646 -0.4759878 -0.7677598 -0.17532511 -0.51903784 -0.8219462 -0.40157586 -0.035203844 0.3355533 0.7282126 0.510323 -0.8017598 -0.2608678 0.750926 -0.9074293 -0.5022462 -0.52687836 -0.81569993 -0.9460118 0.09626383 0.70283365 0.726331 0.577339 0.19115247 -0.5614761 0.60198855 -0.9193974 0.9540863 -0.46225902 -0.46574804 -0.43450937 0.32179677 0.6719147 -0.18040863 -0.20260125 -0.11380186 -0.874621 0.21416302 -0.5065564 0.41148558 -0.64522576 -0.62846595 0.17629759 0.035399392 -0.19169107 -0.66519624 -0.5995269; -0.4856275 -0.23704271 -0.5277152 0.41959038 0.4803737 -0.7227928 0.7943417 -0.29231027 0.9189615 0.48653194 -0.8100154 -0.04531489 0.53653145 -0.6886191 -0.62338483 1.0570725 0.44753492 -0.2675482 0.15489456 -0.5636845 -0.7406883 -0.6825529 0.52842635 1.1275883 0.79171264 0.62222105 -0.30331308 0.61635745 0.64363295 -0.4655163 0.88052315 1.0676225 0.39422342 -0.7069939 0.33655825 1.1398389 1.0652876 1.1812651 0.66662455 -0.32804373 -0.35777894 -0.38232166 0.5951605 -0.37916648 0.5195832 0.03078969 -0.7908263 0.17927946 1.0116887 0.94844055 0.6594108 0.5442582 -0.110629775 -0.53533894 -0.8572512 0.71419287 0.565449 0.6443418 -0.71318966 0.57546496 -0.8745451 -0.2649999 -0.10688482 0.82218146 -0.23903395 -0.82886446 0.93255085 0.6422359 0.6041682 -0.5383918 -0.5492497 -0.5113633 -1.0535129 0.40995738 0.93380255 0.10802696 -0.55128044 0.42758492 0.8760525 -0.64295554; 0.70454293 0.9064765 -0.9545095 0.5486307 1.0209295 0.0067000724 -0.26655847 0.1682176 -0.096266344 0.8111979 -0.07896215 -0.73601323 -0.48476666 -0.15328142 -0.19413063 -0.21832624 0.8380488 -0.7479003 0.53497934 0.09743579 -0.8018976 0.7390746 -0.044794586 -0.3049951 -0.38260224 0.74993384 0.84326047 -0.36514187 0.95171106 0.5561112 -0.38087982 -0.4361815 0.6857235 -0.6626637 0.7486888 -0.5238736 -0.04288412 -0.46861383 -0.05626134 -0.84994215 -0.7921021 -0.7209872 0.7889088 -0.39365372 0.36986387 -0.66725814 0.49552915 -0.09479665 -0.37489036 -0.3377466 -0.3150977 -0.029821 -0.8691549 0.12740134 0.29510048 0.8541199 -0.5696304 -0.6315963 0.2500257 0.26798365 0.07151628 -0.6703281 -0.8822993 -0.034714542 0.10664368 0.1629782 0.57250273 0.13299881 -0.57438755 -0.60442287 -0.7002331 -0.6472461 0.28650066 -0.37565032 -0.3791975 -0.49739483 -0.6165194 0.85110873 -0.24321023 -0.67527074; -0.5339833 -0.62314194 0.6589788 -0.49567324 -0.8751144 0.38699374 -0.030399533 -0.48143527 -0.31318828 -0.67107815 0.57326055 0.7565901 0.24843664 0.47639158 0.31761098 0.19050671 -0.71932876 0.6746678 -0.74930507 0.4142583 0.8151869 -0.74429405 -0.24759053 0.034535404 0.18582557 -0.6889124 -0.6076904 0.20275666 -0.8512177 -0.8187788 0.46818197 -0.0135553805 -0.67707086 0.7577388 -0.4691483 0.12886102 -0.295696 0.33910206 -0.2004739 0.85217416 0.5910313 0.44125497 -1.019218 0.53781503 -0.46873108 0.61751413 -0.2611504 -0.01917789 0.2126371 0.46267375 -0.040240604 -0.22909266 0.8588112 0.4254815 -0.5575263 -0.8564005 0.25098312 0.56349164 -0.29261053 -0.2693828 0.21228932 0.762531 0.73263645 -0.20887443 0.06989082 0.20899537 -0.53737694 -0.44244045 0.6551829 0.98667204 0.5603752 0.9337794 -0.4704808 -0.15122567 -0.11557145 0.37996358 0.7288093 -0.82258236 -0.27966598 0.984567; 0.6865279 0.40185818 0.13276733 -0.30667904 -0.57804334 0.95394075 -0.5901181 0.48131588 -0.6442426 -0.6160615 0.79155016 -0.2638712 -0.36384487 0.41338342 0.8256915 -0.765983 -0.0049521355 0.25355542 0.105160765 0.4338236 0.702724 0.8049289 -0.6956571 -1.1349761 -0.77966374 -0.50754184 0.57268786 -0.61452234 -0.760576 0.3014732 -0.882858 -0.99685127 -0.1025404 0.7925101 -0.28990212 -1.155864 -1.0525974 -1.1946448 -0.75453234 0.16194043 0.5459293 0.4136847 -0.44829273 0.20866391 -0.5569468 -0.33330765 0.72203225 -0.15946822 -0.97551435 -0.939295 -0.4254477 -0.51691765 0.17458743 0.40010294 0.8366667 -0.5896827 -0.33576548 -0.67441493 1.0661899 -0.4221535 0.71011716 0.07568914 -0.15882951 -0.82964444 0.010718961 0.94020563 -0.9776944 -0.7806132 -0.6540934 0.45513296 0.25637233 0.2646807 0.76433635 -0.4986294 -0.9053825 -0.37524757 0.5550115 -0.21807869 -0.5822578 0.638715; 0.25494096 -0.17900486 -0.5786615 -0.0009590174 0.514368 -0.15992157 -0.7971991 0.63618165 -0.13846882 0.6829519 -0.73176605 -0.68859255 -0.6433442 0.7969599 -0.283646 -0.26122147 0.14438248 -0.25292686 0.70662713 0.91131777 -1.3155906 0.77451926 -0.45176104 -0.18410376 -0.26262265 0.9703943 0.11414467 -0.9235422 1.3965306 0.35343266 -0.18581565 -0.51019716 0.7290899 -1.0393609 -0.5926851 -0.37531573 0.42506438 -0.0682711 -0.5789795 -0.6815393 -0.1918886 0.28664678 0.89392567 1.0748127 -0.6522758 0.016588096 0.25553668 -1.0597886 -0.36108232 -0.4420238 -0.9237462 -0.99607754 0.4589253 0.7150743 0.6427843 1.3325671 0.6303578 -0.8303589 0.37532046 -0.96879274 0.43062446 -0.5125204 -0.8931706 -0.09672167 1.2973248 0.08861859 1.1851634 0.43629107 0.46735895 -1.0353551 -0.31317717 -0.9427353 0.44603163 -0.73526067 -0.08310089 0.784912 -1.3405244 0.12288704 -0.3762224 -1.1856389; -0.49895522 -0.39356312 0.39217302 -0.21445721 -0.06852435 -0.8492639 0.3072852 -0.16988511 0.5472829 0.16530348 -0.8274716 0.88819 0.03391606 -0.36301365 -0.49342275 0.8543908 -0.15361291 0.2497481 -0.54788154 -0.256894 -0.24736816 -0.879402 0.58228713 0.78058964 0.11070264 0.6287395 -0.44306284 0.39212522 0.32997936 -0.44952986 0.61525047 0.7984038 -0.75345767 0.16608135 -0.043779857 0.77279764 0.78435165 0.9784026 0.4285082 0.7741169 0.09971482 -0.06590938 -0.412648 0.17780024 0.06461846 0.2593762 -0.2888129 0.061246835 0.98718566 0.9197314 0.24732725 0.082269266 0.112197615 -0.2950316 -1.040688 0.24417566 0.40397084 0.9092499 -0.8851765 0.07375377 -0.9913557 0.37430924 0.7810252 0.6835176 0.063097395 -1.014222 0.4776016 -0.11977815 0.53185046 0.10955936 0.15439647 0.7269301 -0.70019585 0.09624153 0.83026105 0.23656186 -0.3799761 -0.013339042 0.27486715 -0.25055072; 0.5236498 0.72075075 -0.75316346 0.9209593 0.8155793 0.4293492 -0.4526926 0.41624683 -0.50863165 0.9043757 0.4057305 -0.61456925 -0.49458292 0.33758372 0.5082079 -0.51207095 0.9369821 -0.81174713 0.72313577 0.15033253 -0.046388403 0.7414476 -0.30961934 -0.6326739 -0.4168785 0.6711392 0.639228 -0.4055794 0.57641315 0.7076041 -0.34482026 -0.32897094 0.7997919 -0.83212835 0.64600873 -0.6269769 -0.32542765 -0.5470093 -0.30795065 -0.8177513 -0.79973215 -0.7686031 0.46656463 -0.62279344 -0.2774194 -0.85698426 0.2583704 -0.1899158 -0.37312207 -0.5311291 -0.29036784 -0.048205234 -0.71235675 0.4770048 0.5101289 0.6015366 -0.6799875 -0.6126546 0.5085798 -0.09152221 0.47190416 -0.7657627 -0.547899 -0.6261141 -0.12685569 0.25909695 -0.20259191 -0.401273 -0.82490784 -0.5360444 -0.8574308 -0.6855669 0.6009603 -0.18828842 -0.40378714 -0.94382644 -0.4979414 0.7613825 -0.597134 -0.62124; -0.3459876 -0.070524365 -0.37114298 0.52383435 0.6277348 -0.86049104 0.65810114 -0.77714914 0.8827567 0.60694194 -0.7369327 -0.11583914 0.49296626 -0.6276195 -0.9983898 0.8394671 0.15208553 -0.41825047 -0.07576211 -0.6466016 -0.7458167 -0.14665188 0.88711214 0.6851775 0.6617907 0.3362679 -0.43103665 0.423809 0.58868676 -0.4970502 0.7856641 0.81309885 0.71306777 -0.83133465 0.619369 0.6686617 0.8974046 0.7536052 0.9382857 -0.22649118 -0.28669727 -0.58414304 0.2487389 -0.30703738 0.7242291 0.11854094 -0.5746206 0.5443858 0.5541442 0.6497041 0.54663706 0.75470704 -0.27246594 -0.7011444 -0.44283167 0.49142244 0.16868097 0.44490585 -0.89076334 0.8938678 -1.0275487 -0.032567825 0.010084004 0.94331396 0.08051923 -0.9072714 1.0645711 1.1108452 0.29764852 -0.07760058 -0.610143 -0.8442492 -0.5267388 0.77083176 0.99348587 0.14236541 -0.92227256 0.53038293 0.86027557 -0.37851855; -0.61047834 -0.6440516 0.88270825 -0.7260597 -0.87287414 -0.56869596 0.61508983 -0.37832457 0.3350983 -0.5710767 -0.5058414 0.61600876 0.5751601 -0.43933383 -0.3143986 0.44610745 -0.8122448 0.82043207 -0.529286 -0.5694318 -0.22730854 -0.7228374 0.24387719 0.3055772 0.37293524 -0.5168856 -0.8537813 0.38601384 -0.5578307 -0.5206145 0.58537734 0.38118768 -0.6634993 0.5369768 -0.53123724 0.4327758 0.64895684 0.355299 0.56208533 0.7268798 0.7402983 0.52650607 -0.83814555 0.72158974 0.26645866 0.5187609 -0.6314965 0.40084073 0.4330409 0.49841753 0.31223148 0.36113787 0.6169164 -0.4328454 -0.69076246 -0.7657185 0.6635673 0.6098716 -0.6372288 0.18304202 -0.31638804 0.51262826 0.706369 0.4776761 -0.056305196 -0.57650393 0.2723655 0.63198596 0.5663574 0.53151137 0.7930889 0.64258385 -0.6778386 0.545524 0.6286435 0.5762659 0.44171458 -0.727104 0.41616836 0.73019654; -0.33096382 0.045729414 0.56732714 -0.616092 -1.011965 0.74156976 0.45462194 -0.66598237 -0.53441 -0.723801 0.93052256 0.8470142 0.42559922 0.12661824 0.7381575 -0.0514903 -0.36748287 0.7578108 -0.6842081 -0.010554578 0.83892655 -0.78294796 0.2707922 0.10946498 -0.05365942 -0.9244622 -0.2774097 0.6589831 -0.9325785 -0.6695353 -0.036410276 0.024124755 -1.0766425 0.880903 -0.5493317 -0.060619783 -1.0783834 -0.026714887 0.07933833 1.0254328 0.7161686 0.32226554 -1.0230525 -0.051885772 -0.12949537 0.114039816 -0.21710354 0.30980432 0.046995793 0.48848617 0.52055055 0.14756171 0.47436422 0.061962117 -0.9989185 -1.0180372 -0.90135115 0.60063446 -0.29202044 0.08326851 0.28904533 0.6092614 0.9646268 -0.7613656 -0.9914803 0.1542109 -0.8478289 -1.1929649 -0.25112182 1.205579 0.68534595 1.0107429 -0.121121906 0.30096498 -0.23358561 -0.63055885 1.1352031 -0.6866209 -0.48229894 1.2394868; 0.5986886 0.19687931 0.52931213 -0.4167919 -0.6803704 0.74813354 -0.83616066 0.5702459 -1.0167875 -0.5990067 1.0157071 0.3338643 -0.73172927 0.74101377 1.0610327 -0.89888734 -0.38594875 0.36313865 0.0943589 0.8169399 0.91919726 -0.029161273 -0.72500485 -0.79958653 -0.8079989 -0.47636822 0.4883238 -0.47001708 -0.6649411 0.3783685 -0.7424357 -0.8252758 -1.0052772 0.9270947 -0.7214404 -0.9564014 -1.0916107 -0.736032 -0.7092214 0.16996251 0.33048713 0.49555135 -0.63960946 0.035716914 -0.8960912 -0.2509632 0.714997 -0.6704393 -0.9215536 -0.76505584 -0.6623389 -0.77092427 0.37558064 0.67579186 0.5548793 -0.73312616 -0.42147267 -0.3643227 0.71498674 -0.7568497 0.8207307 0.39120626 0.28403026 -1.008 -0.20298864 0.7649563 -0.89310074 -0.9278894 -0.306464 0.38652733 0.48329723 0.7301655 0.7986447 -0.7030044 -0.70703065 0.06549079 0.66391116 -0.5306518 -1.043751 0.6492646; -0.09451444 -0.14784516 -0.47043985 0.33250225 0.7180942 -0.8814554 0.09617142 0.42048407 0.8098173 0.82059246 -0.9917 -1.2391431 0.26174727 -0.4114679 -0.9973061 0.84266716 0.5911689 -0.45877948 0.67577624 -0.41768074 -0.8799175 0.7272995 0.54695106 0.941261 0.7428071 0.736684 0.20245913 -0.039524462 1.0527556 0.074805014 0.8177799 0.9250893 1.2349808 -0.8733057 0.41496786 0.8921116 0.8528023 0.86029464 0.43848673 -0.92147934 -0.52942127 -0.3933756 0.91740525 -0.0067977468 0.41910565 0.049652137 -0.883418 0.18803817 0.8296412 0.51594526 0.1438915 0.40953907 -0.2061773 -0.5259026 0.47951865 0.8886344 0.4630247 -0.54583675 -0.6676499 0.30532733 -0.7189723 -0.68144524 -1.0300419 0.7251274 0.40875676 -0.8115953 1.1022563 0.93175036 0.44196105 -1.2922782 -0.40365165 -0.97270656 -0.5368864 0.2758197 0.91218996 -0.05666777 -0.7923363 0.52578175 1.010359 -1.174724; -0.7319984 -0.27897927 0.32719138 -0.26384637 -0.4532494 -0.64655393 0.6838375 -0.42120066 0.77069473 -0.34307048 -0.8488663 0.42045176 0.42438197 -0.7268161 -0.65008885 0.5788954 -0.1954837 0.49849057 -0.44533288 -0.48597783 -0.12873457 -0.64821714 0.74744725 0.78476137 0.7311456 -0.07792502 -0.41481364 0.4343242 -0.07880869 -0.6672958 0.6979006 0.5542157 -0.4766027 0.25682217 0.1408754 0.8087161 0.5961248 0.8103559 0.5540146 0.5670466 0.55710125 -0.27950945 -0.27360213 0.1595933 0.68667114 0.25906965 -0.46487698 0.42789844 0.8176 0.6558474 0.57921606 0.5930441 0.33300108 -0.47998172 -0.75488853 -0.31316602 0.4366703 0.64344263 -0.8128169 0.7045894 -0.5744506 0.4034864 0.69955117 0.7063971 -0.22164023 -0.8240427 0.60806954 0.72076195 0.67199975 0.22019343 0.43346244 0.38535833 -0.76277024 0.4341376 0.8583362 0.26310396 -0.23388432 -0.3913665 0.57652485 0.27898106; 0.69718796 1.0138652 -0.5935036 0.6962015 0.69364095 -0.38534427 0.2585795 -0.04718518 0.10412223 0.6490574 -0.52064735 -0.6362644 -0.44965944 -0.6284244 -0.47760886 -0.0453504 0.5849301 -0.66714144 1.023948 -0.40773 -0.78514194 0.79850125 0.39131224 -0.017494386 0.045858804 0.8631823 0.9419702 0.042627666 0.59422517 0.7180069 -0.6473211 -0.10406508 0.58035964 -0.5234033 0.7901428 0.065132625 0.44633114 -0.14222601 0.37267053 -0.57864827 -0.6488247 -0.70319605 0.7104511 -0.59523445 0.71316415 -0.7123761 0.15639932 0.23153146 0.14455025 -0.44841632 0.09753304 0.5307699 -0.7672483 -0.29452938 0.44761744 0.6631208 -0.6043986 -0.5329406 0.15992281 0.40697193 -0.46364123 -0.6026418 -0.7561417 0.120023794 -0.43711972 -0.4062121 0.59959775 0.2888221 -0.91321236 -0.89058757 -0.5772187 -0.5534 0.5606841 0.11117462 0.43834135 -0.6428406 -0.71279705 0.7836245 0.29486814 -0.59483004; -0.26377463 -0.28747022 -0.29390287 0.58611786 0.68338156 -0.9326302 0.6130785 -0.74834144 0.94017047 0.5332664 -0.7043374 -0.05759689 0.6878737 -0.9299456 -0.72946113 0.9007236 0.28267133 -0.49051893 -0.07921519 -0.8462058 -1.021068 -0.061673082 0.9288439 0.6232464 0.7056609 0.40816396 -0.293115 0.61031806 0.78837436 -0.30967218 0.6742299 0.6782375 0.6696677 -1.0377154 0.6733146 0.58256304 0.81167835 0.83688956 0.7113889 0.02826614 -0.29060042 -0.444705 0.468515 -0.34073764 0.7823912 0.14563519 -0.62402046 0.59913826 0.8916938 0.5217317 0.922136 0.76580703 -0.2275461 -0.9646352 -0.3665984 0.696314 0.13945615 0.5831156 -0.62010676 0.7478449 -1.0703282 -0.09392173 -0.021490466 0.8197035 0.518031 -0.98268944 0.77728945 0.8078235 0.3238217 -0.15946078 -0.4203161 -0.8207789 -0.7489568 0.862387 0.9134055 0.04070483 -0.734953 0.6400572 0.8447181 -0.4624488; -0.70300275 -0.7174688 0.9541466 -0.6538695 -0.88684833 -0.42060113 0.29454905 -0.20811023 0.26665023 -0.75443727 -0.044230945 0.6892497 0.41002297 -0.21291167 -0.29025275 0.23893023 -0.8483628 0.686157 -0.5356357 -0.021442551 0.6133455 -0.67577696 0.10061878 0.29459903 0.22097723 -0.99129516 -0.65826064 0.3512298 -0.7297038 -0.8562425 0.60171705 0.42975953 -0.62436736 0.9653195 -0.5814691 0.5156817 0.111094475 0.5550068 0.19312479 0.62972337 0.67836004 0.5217758 -0.7849705 0.7100507 0.11617109 0.86333954 -0.28043446 0.35982683 0.466708 0.5450842 0.25131616 0.25797972 0.69014657 -0.22992463 -0.72220176 -0.79754496 0.6119228 0.58529484 -0.39033663 0.106433176 -0.36600554 0.66557723 0.6216071 0.3132429 -0.1678065 -0.38562486 -0.037222862 0.35592085 0.55271214 0.74801713 0.79794145 0.641273 -0.4710835 0.3056366 0.47146505 0.7447318 0.73329794 -0.9616938 0.16895407 0.8490528; -0.7335724 -0.49125278 -0.16352075 0.54985374 0.4031706 -1.034539 0.91158396 -0.6188576 0.99313676 0.38960195 -0.99889094 -0.074476585 0.3754249 -0.6248153 -0.8749079 0.8121166 0.40506873 -0.29457518 -0.1917138 -0.5152061 -0.59566784 -0.63117266 0.77839446 0.7543397 0.57438755 0.6034967 -0.33709273 0.42535338 0.7013536 -0.4392813 0.99279976 0.86067605 -0.037762176 -0.5266158 0.5878852 0.7728466 1.1126877 1.0351288 0.4731357 0.083722614 -0.57668865 -0.47708756 0.22118454 -0.012968243 0.44287485 0.38063207 -0.7297538 0.23972642 0.9686655 0.9141 0.59730023 0.47391793 -0.009337013 -0.48501766 -1.133582 0.5514191 0.44963038 0.5607709 -0.9829066 0.43464735 -0.73563445 -0.38362145 0.0172237 0.72609687 -0.031126706 -1.0704602 0.8852308 0.7318932 0.6614266 -0.45606753 -0.27504468 -0.46574095 -0.8613032 0.71421975 1.065896 0.40403983 -0.931013 0.26744854 0.78680754 -0.70301324; -0.030976592 0.2240726 0.889443 -0.5752102 -0.8998592 0.6630988 0.43263006 -0.6580557 -0.45744628 -0.7643293 0.76467466 0.86305594 0.6530642 0.017410243 0.71812636 -0.21263438 -0.4985936 0.58974767 -0.5801891 -0.17978834 1.0417659 -0.83742714 0.08849839 -0.0686666 0.25467953 -0.73069584 -0.2790711 0.52139467 -0.8353758 -0.53814244 0.0014118496 -0.0042610867 -0.7232528 0.8557858 -0.29133624 0.13426955 -1.1165856 -0.020750912 0.018265015 0.88593525 0.6587022 0.5489966 -1.1495076 -0.36483186 -0.4108535 0.2193241 -0.28182444 0.6095562 0.093815014 0.43495587 0.6315395 0.1999646 0.16484182 -0.114836596 -0.517756 -1.1887331 -0.9653559 0.7500053 -0.15209205 -0.2692994 0.41463816 0.61797184 0.92673296 -0.66494304 -0.8637802 0.3265479 -0.88089365 -0.90804636 -0.551448 0.95412946 0.5359873 1.0122917 -0.30888247 0.34029174 -0.27444923 -0.6780948 0.8754168 -0.5701848 -0.5138746 1.01372; 0.8449846 0.68885154 -0.6856501 0.7159356 1.0025738 0.119340315 -0.44027597 0.3869009 -0.11061732 0.7755334 -0.28001085 -0.5716869 -0.44453153 0.17371781 -0.006033834 -0.43203703 0.6665935 -0.71818423 0.75538415 0.063153945 -0.83711255 0.6030287 -0.19936912 -0.18794984 -0.4575694 0.6396215 0.6259316 -0.25232056 0.8913398 0.6895241 -0.41085598 -0.57972187 0.6893601 -0.87874365 0.7321933 -0.2619095 -0.055119783 -0.3784846 -0.17076938 -0.7915407 -0.7887544 -0.54499704 0.87816167 -0.40635255 -0.026708625 -0.6240135 0.3924595 -0.44538724 -0.34219837 -0.27637836 -0.31936228 -0.12241017 -0.86064714 0.21161033 0.8270236 0.7815015 -0.6839762 -0.5764762 0.49735638 0.09734151 0.17048241 -0.7580662 -0.90974337 -0.059506208 0.30110204 0.35183957 0.43559176 0.11266464 -0.6488934 -0.9221579 -0.671117 -1.0178638 0.68603635 -0.131954 -0.4791366 -0.6095468 -0.8645273 0.74973273 -0.2718432 -0.7098418; 0.55660796 0.47230774 -0.15128516 -0.09394829 0.07361338 0.7364853 -0.50547206 0.3546533 -0.6634872 -0.35886377 0.5565597 -0.75576824 -0.28193912 0.68376595 0.72828025 -0.7691279 0.1834971 -0.012267966 0.57203484 0.620091 0.22705878 0.9074065 -0.71051174 -0.91975814 -0.48355785 -0.4103785 0.46926913 -0.53728557 0.065540425 0.48546576 -0.7383557 -0.64923155 1.0535209 -0.10020556 -0.0823726 -0.70072263 -0.7700867 -0.8064251 -0.54417473 -0.6833368 -0.2652036 0.17061353 0.572597 0.1787293 -0.61967564 -0.35186684 0.6451504 -0.39896894 -0.94714254 -0.6947024 -0.696438 -0.7209933 -0.15501279 0.4351041 1.2069573 0.012731613 -0.18374683 -1.2259028 0.6937544 -0.31024238 0.6316768 -0.14038181 -0.88832676 -0.70801586 0.49136508 0.9724204 -0.5268468 -0.13043843 -0.28609702 -0.3496095 -0.19809705 -0.4813546 0.93522173 -0.5062293 -0.9760269 0.098130025 0.34452033 0.13225594 -0.30878964 -0.2765813; 0.86616033 0.24136797 0.5411099 -0.55170375 -0.5850982 0.97125965 -0.41744277 -0.034152947 -0.79760885 -0.5934113 0.6597517 0.7499054 -0.2850644 0.38065833 0.8853339 -1.0508759 -0.61149436 0.32708675 -0.4449665 0.49070552 0.6582355 0.23750772 -0.5589422 -0.783547 -0.83266073 -0.5163688 0.3438782 -0.4193773 -0.83684355 0.3183827 -1.0903167 -0.7360687 -0.7464388 0.81033087 -0.3999989 -1.0915174 -1.0564069 -0.84181976 -0.6031342 0.44763204 0.70146346 0.7037871 -0.8465958 0.26682347 -0.4187086 0.06798718 0.832932 -0.066534944 -1.0560077 -0.693641 -0.45807904 -0.3149423 0.50442123 0.43960452 0.9511635 -0.5801428 -0.52374303 -0.92527837 0.81476855 -0.5646558 0.93135774 0.43417805 0.6542389 -0.6138049 0.046008952 0.7739395 -0.9701426 -0.98641515 -0.71707916 0.77488136 0.49135965 0.6431647 0.92961705 -0.36180463 -0.81610733 -0.1390569 0.7415627 -0.4614705 -0.7605867 0.6690682; 0.69490105 0.6239795 -0.43944022 -0.1479693 0.07538129 0.99544436 -0.47245786 0.5560556 -0.64470804 -0.1736171 0.86026764 -0.4257057 -0.6977752 0.6089421 0.86956555 -0.6771501 0.33006403 -0.2576848 0.24003144 0.7348601 0.37751663 0.52736646 -0.86949795 -0.9064849 -0.62398213 0.0929539 0.5493107 -0.66653794 0.13926254 0.7012515 -0.6565706 -0.84499234 0.594272 -0.108164534 -0.19845584 -0.81850255 -0.89651585 -0.7690776 -0.63920444 -0.5710861 -0.20202559 0.3356064 0.32772753 -0.0021242076 -0.8174153 -0.27926686 0.73594224 -0.39815766 -0.8203547 -0.85430807 -0.5207669 -0.6545094 -0.43387237 0.47745594 0.8164433 0.022320464 -0.31645373 -0.6182691 0.6183299 -0.45013613 0.7331487 -0.4013149 -0.5994688 -0.77580804 0.33913273 0.8187783 -0.6919921 -0.5665126 -0.40493223 -0.17994237 -0.051256225 -0.2134903 0.5970467 -0.53795576 -0.95006555 -0.41339698 0.5705466 0.32938644 -0.5396826 -0.06298098; -0.6016924 -0.7851253 0.68349427 -0.5639397 -0.64533 -0.36961532 0.4560048 -0.38311785 0.17400755 -0.94219226 0.1487756 0.8906712 0.4535399 -0.05017642 -0.057262834 0.44004864 -0.7911381 0.85885054 -0.771858 -0.33044967 0.91916454 -0.57829607 0.1496937 0.250978 0.35232407 -0.80371803 -0.54375225 0.33506152 -0.95197266 -0.6768237 0.5050174 0.40702257 -0.6820392 0.8150734 -0.6159387 0.30205688 0.29433087 0.49287778 0.35673225 0.7874377 0.6890934 0.70840067 -0.79296744 0.34574133 -0.12102197 0.7434585 -0.5827361 0.4553148 0.55318594 0.57192546 0.33037767 0.114439644 0.6317604 -0.353895 -0.8082883 -1.0140383 0.38680494 0.64190906 -0.27875707 0.29500332 -0.41020548 0.72867024 0.5953357 0.11586664 -0.39540455 -0.40108576 -0.25229135 0.23789355 0.6619112 0.777431 0.72993594 0.8781559 -0.5512991 0.26415202 0.36811247 0.48789623 0.9620909 -0.70190156 0.4628393 0.9588796; 0.8196462 0.59606415 -0.6788441 0.7001402 0.6501986 0.5229011 -0.53433216 0.51801676 -0.4099428 0.72285426 0.4634134 -0.5044903 -0.46752813 0.4515529 0.39254043 -0.4090654 0.5514022 -0.6942518 0.57757515 0.53784806 0.44898295 0.8675683 -0.57214063 -0.33265418 -0.4628975 0.496059 0.56937534 -0.51802564 0.5196105 0.6238374 -0.5499477 -0.63767743 0.55576515 -0.7649627 0.20261683 -0.7840334 -0.3294968 -0.64516276 -0.40910545 -0.498102 -0.81523436 0.01166157 0.8149388 -0.4798051 -0.56292915 -0.8184913 0.526367 -0.36107907 -0.48152846 -0.589686 -0.63117385 -0.61262393 -0.83883154 0.40733874 0.5857569 0.69738376 -0.7585831 -0.65481097 0.2903158 -0.49012583 0.34900492 -0.61754495 -0.56494343 -0.45681936 0.35511276 0.37513012 -0.6421751 -0.37543696 -0.5910726 -0.7488647 -0.70105636 -0.74864614 0.59099406 -0.44353983 -0.38770863 -0.58633727 -0.0008563983 0.6564527 -0.30980283 -0.65139025; 0.51564 0.5907961 -0.6721562 0.7178223 0.74429905 0.39936754 -0.30373153 0.33025005 -0.5300255 0.6355769 0.23838286 -0.7798223 -0.36014587 0.34680808 0.15011446 -0.29268432 0.9100605 -0.9152153 0.50678027 0.43920234 -0.68028665 0.6389 -0.5352369 -0.5952059 -0.38592896 0.7794514 0.58397746 -0.54659146 0.71842414 0.6271238 -0.59895515 -0.4055238 0.8454027 -0.8318642 0.64905083 -0.43961084 -0.2543655 -0.3178792 -0.19977427 -0.8292218 -0.8281553 -0.4090822 0.5806791 -0.47997242 -0.32529038 -0.59416014 0.53270745 -0.37735334 -0.3922855 -0.2684744 -0.47312906 -0.29867384 -1.0231551 0.5123666 0.47351468 0.8786552 -0.89400816 -0.7543624 0.46313056 -0.2631031 0.19940296 -0.79060066 -0.7819741 -0.32511747 0.035731513 0.5441979 -0.13342717 -0.3251108 -0.75058085 -0.5347014 -0.9635876 -0.8638669 0.65763223 -0.29692006 -0.31714088 -0.8647721 -0.60551584 0.75194556 -0.499542 -0.55520725; 0.5827709 0.4259885 -0.39746597 -0.24785428 -0.4882476 0.8224648 -0.5374094 0.24004008 -0.71893007 -0.4272525 0.77201027 -0.5340693 -0.5677191 0.6861578 0.7767489 -0.6415841 0.21898894 -0.41530037 0.41725123 0.64881665 0.87339324 0.7689967 -0.803918 -0.98777366 -0.7688532 -0.4305833 0.66291237 -0.61758226 -0.77807415 0.6780204 -0.5995947 -0.71421856 0.67616755 0.44367874 -0.08691298 -0.9785082 -0.64198095 -0.93008846 -0.7098296 -0.43555024 -0.12661189 0.16739231 0.18322544 -0.2652454 -0.5324285 -0.44326222 0.63116336 -0.19297314 -0.60589457 -0.6737686 -0.7710265 -0.6191494 -0.37549463 0.54861176 0.87874097 -0.50019795 -0.59280866 -0.74877024 0.7654411 -0.54256785 0.74946946 -0.39974064 -0.62640494 -0.7797965 -0.012936889 0.9057031 -0.8890086 -0.54110146 -0.39000645 -0.27331212 -0.061254647 -0.10104222 0.8956497 -0.49931586 -0.69763994 -0.20877722 0.81341046 0.13427606 -0.82544494 -0.00205372; 0.501597 0.45356745 0.0850087 -0.39142242 -0.44951925 0.9588689 -0.41951263 0.25638583 -0.86155194 -0.43465143 0.7099093 -0.33793092 -0.3590309 0.57290447 0.6661085 -0.8815962 -0.19946466 0.2558371 0.33014858 0.46252048 0.80225724 0.91621023 -0.47981837 -1.0591246 -0.7257534 -0.66918087 0.3986203 -0.3834654 -0.69765824 0.39772904 -0.9047799 -0.77826446 -0.08891843 0.512735 -0.21115048 -1.1263794 -0.7778022 -1.1523697 -0.5288149 -0.18243924 0.17955554 0.55484486 -0.40053588 -0.064589664 -0.3425206 -0.46215472 0.6707385 -0.053593095 -0.8823844 -0.8703165 -0.4488624 -0.45820084 0.008542849 0.6141467 1.0492021 -0.8329054 -0.66431373 -0.7383366 0.88285506 -0.36241588 0.8409129 0.21890168 -0.1950033 -0.97028154 0.023027007 0.84808636 -1.0106002 -0.72880036 -0.37941203 0.36483088 0.17126496 0.4372748 0.9947893 -0.5907424 -1.1489093 -0.47014418 0.8200685 -0.13263254 -0.6152536 0.5854472; 0.35648125 -0.077098325 -0.81024414 0.41827917 0.82089186 -0.45993248 -0.40040743 0.46549976 -0.036296967 0.9339807 -0.6854317 -0.9790965 -0.29021955 0.30518568 -0.5841972 -0.2044779 0.35488233 -0.78379554 0.8450586 0.17425491 -1.0040833 0.48597595 -0.42894936 -0.23882023 -0.35204762 0.76340497 0.45229828 -0.44307694 0.9226269 0.46716362 0.072481655 -0.2527592 0.6979915 -0.98243713 0.16060871 -0.41085064 0.81751883 -0.1373889 -0.078333996 -0.6485437 -0.77608347 -0.6248488 0.7428095 0.5926965 0.01761395 -0.15297726 0.5705832 -0.5133245 -0.21768191 -0.38147244 -0.6997575 -0.5045699 -0.015372951 0.15434621 0.79703814 1.1599011 1.0216987 -0.69209135 0.25600794 -0.058101553 0.111232914 -0.8455392 -0.79080474 -0.102447145 0.82916385 0.27583137 1.2925084 0.68327695 0.4236993 -0.9077295 -0.52595276 -0.71999997 0.2251235 -0.466783 -0.0873245 0.9024722 -1.2282028 0.65670556 -0.20476814 -1.0044528; 0.76417166 0.2636236 -0.2930572 -0.15427226 0.03020785 0.7235175 -0.5413144 0.41625634 -0.600679 -0.2121725 0.84781706 -0.8352014 -0.26307803 0.33253753 0.87219024 -1.0198163 0.1279028 -0.21312815 0.70168585 0.29703516 0.44697672 0.70750326 -0.4834904 -0.7783775 -0.39405718 -0.5647072 0.5082096 -0.25834328 -0.4185806 0.46303445 -0.61293167 -0.7148471 0.5754184 -0.07375173 -0.15568666 -0.9373088 -0.5778575 -0.9523166 -0.5586286 -0.7990428 0.06751194 0.049473587 0.5323062 -0.17174803 -0.5258623 -0.4517118 0.616404 -0.33859456 -0.88022524 -0.9153591 -0.37759322 -0.4782662 -0.29049972 0.37435916 1.0609418 -0.17035648 -0.17517745 -0.7839621 0.8605331 -0.24757004 0.8966309 -0.22006066 -0.6665433 -0.6982934 0.2644511 1.0302391 -0.7373242 -0.33273947 -0.30570605 -0.2584846 0.122294605 -0.2810968 0.85561216 -0.32950273 -1.0423197 -0.16056813 0.57711446 0.0775591 -0.34511387 0.082829036; 0.3346585 0.9595056 -0.54155236 0.67678535 0.30902192 -0.15802673 0.91187006 -0.8121783 -0.05733868 0.6298959 -0.010187528 0.34948805 -0.2764722 -0.80815154 -0.2157346 0.24925327 0.8423084 -0.7427423 0.55853146 -0.800207 -0.026748348 -0.5199653 0.9966617 0.086921476 0.15722848 0.5591221 0.7107306 0.80551505 0.16326235 0.7976231 -0.6407482 0.23313272 -0.10770349 -0.056673504 0.7226959 0.36075088 0.16691859 0.3226264 0.92229646 0.29754108 -0.4934615 -0.6839293 0.12758833 -0.7811379 0.741297 -1.0405186 -0.1732313 0.9156163 0.39966878 0.20229657 0.99882907 0.7281989 -0.6297238 -0.76264006 -0.90426534 -0.048566055 -0.977938 0.7510839 -0.21270128 0.8976232 -0.3709313 -0.6882659 0.14101629 -0.030734008 -0.9263021 -0.2087333 -0.08481262 -0.0054050377 -0.6495664 0.23834558 -0.42645344 0.02563024 0.16237476 1.0620874 0.21701612 -0.8367175 -0.10129703 0.53505677 0.033716835 0.2234995; -0.14292927 -0.2235053 -0.6169666 0.5802109 0.816635 -0.6515265 0.7536525 -0.49216095 0.86265194 0.7723102 -0.74942935 -0.2750913 0.83634967 -0.6057836 -0.9368182 0.64533234 0.3788869 -0.3437456 -0.102514364 -0.6958897 -0.9470887 0.03315605 0.73299104 0.9232673 0.96632725 0.6422961 -0.45125097 0.7487633 0.77858764 -0.27934423 0.5925198 0.61835796 0.9152495 -0.89188236 0.4129944 0.8133734 0.9374808 0.8074558 0.75264037 -0.45094463 -0.48621663 -0.6519847 0.4892885 -0.29280218 0.666579 0.2013068 -0.7539238 0.62987643 0.794954 0.6471066 0.7502728 0.61510515 -0.5116064 -0.85868955 -0.28129187 0.5964285 0.32710052 0.04107079 -0.93017507 0.7361341 -0.7222895 -0.4071696 -0.37003103 0.71669835 -0.15009508 -0.9175713 0.90965885 1.0801408 0.39435157 -0.09230229 -0.56079304 -0.74508387 -0.7039853 0.73852414 0.711052 0.070086814 -0.61952704 0.4573858 0.99139506 -0.49451855; -0.82011366 -0.53162324 0.76542544 -0.94401956 -0.94893676 -0.5018821 0.36852425 -0.52387005 0.5749485 -0.6579275 -0.36146617 0.6113942 0.4822868 -0.50805366 -0.43022144 0.2694574 -0.9016718 0.60720766 -0.5017624 -0.23785046 0.5997083 -0.53009784 0.25151503 0.59129745 0.3260571 -0.83053863 -0.6294912 0.6216531 -0.65211415 -0.6481941 0.6018983 0.44708583 -0.79807836 0.7148937 -0.63543844 0.24218662 0.21048675 0.58571553 0.51397204 0.6098478 0.733338 0.47761875 -0.6026265 0.447737 0.16661239 0.72241604 -0.35401753 0.3535945 0.5513398 0.36976674 0.37561992 0.36795914 0.97373533 -0.16248623 -0.62037736 -0.7124073 0.7715165 0.66597205 -0.44919637 0.35797524 -0.35936382 0.93237776 0.60781145 0.25663093 -0.34329212 -0.4054297 0.2587027 0.24563676 0.69170517 0.7518451 0.9372775 0.8571153 -0.7345276 0.48435193 0.23774898 0.7208501 0.80495095 -0.6468847 0.35067126 0.5870979; -0.78929937 -0.7347063 0.9124938 -0.83625144 -0.80082554 -0.2267048 0.35286403 -0.52384436 0.23487408 -0.7796834 -0.02428102 0.9249588 0.4577568 -0.2715416 -0.20136939 0.34104902 -0.65743196 0.77939713 -0.6997186 -0.38708636 0.6720136 -0.6555924 0.3675153 0.3591708 0.45889375 -0.707246 -0.68289673 0.4410559 -0.9944662 -0.708093 0.44443643 0.4245832 -0.64843106 0.9522571 -0.6954338 0.32313502 0.1412556 0.24276242 0.4013681 0.9339413 0.80885184 0.44167507 -0.6606193 0.5422403 0.20356263 0.6132026 -0.28666764 0.5337784 0.43077755 0.34527153 0.3874377 0.26811093 0.7927106 -0.29173276 -0.6523459 -0.7535542 0.48125514 0.77139974 -0.2850203 0.16879071 -0.2712424 0.9070307 0.7005396 0.47384447 -0.5413758 -0.36451066 -0.10889105 0.049320657 0.69704115 0.743738 0.9641392 0.81356025 -0.72541374 0.5616244 0.27607512 0.49201083 0.75386894 -0.901284 0.29449505 0.88661164; -0.7489715 -0.3731015 0.46622106 -0.27562478 -0.43944445 -0.53718275 0.52865154 -0.030809145 0.16319495 -0.089389876 -0.3587536 0.88815886 -0.12064237 -0.22488286 -0.3087569 0.7773975 -0.28436103 0.41557214 -0.7851635 -0.18357827 0.33061594 -0.7985046 0.6070066 0.64340675 0.110018894 0.22002017 -0.502234 0.23538402 -0.045059208 -0.43564963 0.47508937 0.94684047 -0.6257437 0.38892004 -0.094816945 0.72060317 0.56621796 0.7383079 0.19011177 0.73907614 0.3953037 -0.094423704 -0.81370604 -0.05019196 0.33763728 0.18140045 -0.39680707 0.10785848 0.96342933 0.7734038 0.50069153 0.035994597 0.3390161 -0.3763442 -0.78860843 -0.007181013 0.19878879 0.8161701 -0.73701453 0.041759107 -0.67080784 0.56643605 0.59944093 0.6190839 -0.20608708 -0.9888832 0.16390052 -0.37106052 0.45536104 0.15806773 0.18982169 0.8709956 -0.88255847 0.19847512 0.90854305 0.31095994 -0.24216194 -0.48650485 0.17592083 0.25620562; 0.41574758 0.64927083 -0.76519096 0.75934106 0.73172027 -0.4128775 -0.06529031 0.098352 0.12397883 0.7174221 -0.7071111 -0.8602548 -0.38532823 -0.21010888 -0.3838168 0.29960105 0.4656424 -0.6243921 0.9396533 -0.4233726 -0.83861446 0.5951011 0.13516611 0.20193665 0.041154522 0.58982295 0.60030043 -0.26780605 0.8781937 0.56520075 -0.2623207 -0.19393422 1.0204934 -0.7166488 0.48670417 -0.10253281 0.6669449 -0.021786312 0.27195567 -0.9825531 -0.67105365 -0.64962655 0.99374706 -0.29572275 0.3649329 -0.622465 0.13755913 0.08532004 -0.10177556 -0.4851794 0.023848677 0.23613085 -0.7637071 -0.331583 0.2599911 0.84162796 -0.17710614 -0.5915587 -0.033054177 0.24596328 -0.30672652 -0.6860904 -0.8106825 0.5162548 0.025738042 -0.18490055 0.61098224 0.73479724 -0.48815882 -0.6756931 -0.61261034 -0.8567745 0.2673332 -0.011120624 0.4276719 -0.31747755 -0.72044665 0.6009343 0.39404592 -0.88784975; -0.6558702 -0.20772249 -0.4032604 0.32233918 0.4008384 -0.99320567 0.7520885 -0.3225152 0.5042305 0.685532 -0.9173709 0.4992999 0.21945293 -0.76231337 -0.81558955 0.9315973 0.16649634 -0.16993576 -0.42784902 -0.773003 -0.56947136 -0.9120447 0.80868113 0.9735296 0.39310023 0.6343776 -0.37710807 0.6900206 0.6419935 -0.5446725 0.5198769 0.8970041 -0.61759347 -0.2336337 0.2591443 0.99887705 0.8747801 1.163083 0.5757116 0.4280255 -0.34082028 -0.28997016 -0.07463201 -0.2568556 0.62542206 0.04933556 -0.7505484 0.16014291 0.8924143 0.67684233 0.8427453 0.38667604 -0.045919828 -0.4294728 -1.1414592 0.5079393 0.3179814 1.1804608 -0.9681039 0.47607124 -0.8613857 -0.1306331 0.41099176 0.9365727 -0.47689292 -0.77936584 0.7742202 0.4713755 0.2067219 0.11778121 -0.20634025 -0.11135804 -0.9472593 0.42070124 0.9513478 0.17230752 -0.4566888 0.16236159 0.50844187 -0.4494498; 0.2023299 -0.23360668 -0.7279848 0.5449483 0.6207053 -0.7028822 -0.30006373 0.71113145 0.2712312 0.72933686 -0.9976099 -0.9736674 -0.5751771 -0.11516923 -0.51654494 -0.17507207 0.39317536 -0.6469718 0.78746486 0.0012166968 -0.95383334 0.5311618 -0.30907694 -0.29166743 -0.054874483 0.85531676 0.55736524 -0.69525075 1.0105985 0.5787113 0.10608797 -0.3321555 0.8154988 -0.9825383 0.37022203 -0.111825965 0.85403275 -0.22975934 -0.2576508 -0.7882052 -0.52902085 -0.5742333 0.85729593 0.19459027 -0.033946257 -0.058850445 0.18253204 -0.38707927 -0.15926123 -0.4609597 -0.33785248 -0.28898814 -0.40809038 -0.002683711 0.6516162 1.1671162 1.2256563 -0.632808 0.1227932 0.006477422 -0.13764192 -0.69317263 -0.6482123 0.48049885 0.66489357 0.026420655 1.0223767 1.0877877 0.4872035 -1.1045713 -0.44596386 -0.7196118 0.21480423 -0.63715875 0.3335873 0.73214895 -1.1242425 0.6172429 0.38177243 -1.1581539; -0.6212963 -0.27494922 0.9249043 -0.57374364 -0.7019604 0.13280521 0.6269721 -0.48176694 0.11299592 -0.9107346 0.46241933 0.6467382 0.41690764 -0.44780725 0.3621713 0.32667637 -0.6527503 0.8342226 -0.6649584 -0.18825269 1.0257729 -0.6038881 0.3939362 0.4366065 0.51316667 -0.9009025 -0.60489637 0.5716686 -0.7563018 -0.46395335 0.43656525 0.3392136 -0.72056794 0.91364306 -0.55996424 0.26446503 -0.13708502 0.40149948 0.297097 0.6646199 0.6598646 0.47329944 -0.78312063 -0.286959 0.22666861 0.42632562 -0.31167346 0.41426653 0.47810474 0.33948094 0.6184936 0.31547403 0.28839543 -0.39589426 -0.65097064 -0.74328405 -0.62515926 0.6534193 -0.21345153 0.39218786 -0.02661885 0.80747604 0.85922754 0.051550694 -0.72535247 -0.10837566 -1.1443766 -0.54718304 0.10307426 0.8921054 0.7067871 0.88447607 -0.5225792 0.37182167 0.07788768 -0.8856995 1.145228 -0.7773834 0.3440618 0.95315635; -0.6109257 -0.48302907 0.8694626 -0.71873516 -0.93795013 0.27644858 0.27001435 -0.5886165 0.06590216 -0.73577666 0.5338313 0.8149487 0.40159488 -0.09295442 0.34506845 0.29277793 -0.5313607 0.59051716 -0.7185747 -0.06116365 0.814371 -0.71484876 0.030523926 0.2530199 0.0892316 -0.76592386 -0.7757397 0.54035884 -0.9081858 -0.6075608 0.14583789 0.35753673 -0.8283248 0.8832306 -0.6920977 0.22788566 -0.527209 0.21031845 0.13826871 0.8008071 0.75478786 0.43829152 -0.9827173 0.1866747 -0.11141401 0.7492337 -0.26124197 0.118176125 0.17676729 0.32759345 0.52259326 0.04273396 0.70330477 -0.06560988 -0.66813225 -0.912242 -0.08948369 0.5125968 -0.41121903 -0.2641121 0.08941842 0.7886127 0.7369603 -0.23968132 -0.36071238 -0.17417502 -0.959843 -0.7981154 0.429778 0.8585119 0.72479314 0.8654302 -0.74768883 0.22984432 -0.11403856 0.16086955 0.83797324 -0.641368 -0.23462977 1.0476215; -0.5311547 -0.6442253 0.6619502 -0.6841439 -0.7718148 -0.49329093 0.28167337 -0.29043832 0.5291396 -0.6144134 -0.105752714 0.6500396 0.57519877 -0.3647866 -0.37580487 0.39454716 -0.8564025 0.67956835 -0.6561164 -0.23483576 0.55836916 -0.73678327 0.45553383 0.37241971 0.23497888 -0.92841864 -0.6363626 0.47637805 -0.5947214 -0.79031056 0.31151682 0.5214051 -0.5841332 0.84983885 -0.5137867 0.3488143 0.40074384 0.29673404 0.47509587 0.8895093 0.66921175 0.63663197 -0.7641117 0.6474617 0.35639694 0.67923415 -0.3238831 0.3250976 0.517158 0.507575 0.49804235 0.18289481 0.90743023 -0.30307117 -0.7997137 -0.5781004 0.585788 0.72994447 -0.28713876 0.3750586 -0.36567867 0.7324087 0.6680512 0.30986315 -0.06043975 -0.33927304 0.0534205 0.19495395 0.77048826 0.6416527 0.66519207 0.77336305 -0.53090996 0.25454122 0.50750417 0.5959956 0.60853964 -1.0043582 0.43673435 0.8369485; -0.20951216 -0.28250703 0.24507955 -0.65950143 -0.8955254 0.25027668 -0.75768936 0.6164711 -1.0080214 -0.43415272 0.5985627 0.48079878 -0.6706285 0.72699976 0.68659294 -0.64702594 -0.48594138 0.3079707 0.062661216 0.76380986 0.89366144 -0.098135285 -0.8701143 -0.39800304 -0.9546763 -0.0540886 -0.12885928 -0.82445604 -0.43331718 0.08248771 -0.23937383 -0.53894824 -1.010862 1.2883244 -0.7428836 -0.27699253 -0.49928057 -0.35012957 -0.8379572 0.43907958 0.6130719 0.72612935 -0.38006622 0.5523218 -0.7219832 0.18615067 0.7947681 -0.8297465 -0.27304125 -0.4506709 -0.91349506 -0.7718287 0.53758174 0.8941823 -0.022167642 -0.27839917 0.28074554 0.10003901 0.71370983 -0.6697144 0.781923 0.29813325 0.39597136 -0.83405626 0.17102998 0.5486091 -0.40207782 -0.86466116 0.27987933 -0.36177823 0.78924805 0.88954186 -0.028588027 -0.7966656 -0.43921095 0.1377885 0.45547447 -0.69837826 -0.8286436 0.08985878], bias = Float32[0.08739938; -0.78666997; -0.08160888; 0.43809438; -0.32840016; -0.21678749; -0.39615884; 0.16762954; 0.6059423; 0.6999928; -0.24307038; -0.2678679; 0.64132655; 0.51251245; -0.7754097; -0.22913311; -0.016536; 0.6269661; -0.06260145; 0.29520205; -0.10191003; 0.81656826; -0.31708497; -0.82757646; -0.44146642; -0.5293795; 0.62065065; 0.69751686; 0.28999814; -0.09924385; 0.051571656; 0.34632298; 0.7785444; -0.42929164; -0.30022874; -0.12530643; -0.5272405; 0.8360539; 0.38885093; -0.696087; -0.014287594; 0.3304617; 0.7538184; -0.0796484; -0.6426145; 0.39943376; -0.84214056; -0.42851093; 0.45487112; 0.8222032; -0.7310136; -0.6365457; -0.37705982; -0.7864305; -0.17506665; -0.77581924; 0.41290227; 0.1002248; 0.598586; 0.69455624; 0.7056685; -0.14661595; 0.43554085; 0.3145138; 0.6991337; 0.8063163; -0.19620597; 0.72639555; -0.31277287; -0.8588804; -0.3092032; -0.17138465; -0.5576463; -0.40897265; -0.68198687; -0.38114077; -0.066844955; 0.18502118; -0.30430198; 0.5149578]), layer_3 = (weight = Float32[0.2820152 0.6862601 -0.97395164 0.3667874 -0.58193624 -0.40737447 0.6733983 0.27904594 -0.51119286 -0.73501414 0.37356955 0.43265405 -0.60293365 -0.7766419 0.63730305 -0.32351765 -0.32224104 -0.61770636 -0.565764 0.15026051 -0.7163999 -0.6176534 -0.12890115 0.33603042 0.86262 0.43133217 -0.58671373 0.21168286 0.5326461 0.50609374 -0.32236522 0.08197421 -0.7617143 -0.11745367 -0.19789572 -0.3295465 0.5736469 -1.0946361 -0.5266321 0.71226597 0.40692508 -0.4855552 -0.63676083 0.5033993 0.5439765 0.34618372 1.0338515 -0.24339227 -0.97118104 -1.0265747 1.413585 0.7688636 0.48827153 1.0094672 -0.32119623 0.64887744 -0.70668995 0.33775473 -0.5841534 -0.613859 -0.84354204 -0.33633342 0.12554859 0.2561811 -0.68562746 -0.54930115 0.6972426 -0.59499955 0.45310634 0.84612614 -0.25923944 -0.2939674 0.49500573 0.51152307 0.6043446 0.7899229 -1.0889921 -0.64230585 -0.16788074 -0.82448304; 0.03867912 -0.22364046 0.9225471 -0.24688841 -0.23962434 0.2479605 0.23069227 -0.120100304 0.4329253 0.5182784 -0.17749456 0.01675596 0.89225155 -0.5201739 -0.4149425 0.11338446 0.29302827 0.9966809 -0.337894 0.25839195 0.86667866 -0.06245001 0.0031020637 -0.19650516 -0.5730333 1.0195554 0.835335 -0.18245904 0.100754514 0.7396284 -1.0223591 0.42872038 0.46448836 0.3101574 0.17679591 0.3751419 -0.8815672 0.57733935 0.9028108 -0.10984776 -0.11824807 0.029426813 0.19320272 0.7409799 -0.7335923 -0.282366 -0.52543736 0.14496994 0.65278906 0.44543532 -0.3960562 -0.58325195 -0.2233574 -0.63875705 0.2863889 -0.19688573 -0.6133099 -0.053394765 0.9212761 -0.21311316 0.52939135 -0.07488946 0.09282647 -0.25959712 0.5189006 0.05990865 0.53376937 0.56851375 -0.5757997 -0.45419222 0.2822518 -0.24037807 -0.74438787 -0.07527234 -0.34979442 0.36934564 1.0744332 -0.19249868 0.27226523 -0.27680066], bias = Float32[-0.72231174; -0.011232903]))

loss_neuralode(p_trained_high)